<?php

// CORS Headers
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
header('Content-Type: application/json');

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include database configuration
require_once '../config/database.php';

// Initialize database connection
try {
    $database = new Database();
    $pdo = $database->getConnection();
    if (!$pdo) {
        throw new Exception('Failed to get database connection');
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database connection failed: ' . $e->getMessage()]);
    exit();
}

// Token validation function
function validateToken($pdo, $token) {
    if (!$token) {
        return false;
    }
    
    try {
        $stmt = $pdo->prepare("SELECT id, email FROM users WHERE remember_token = ? AND status = 'activated'");
        $stmt->execute([$token]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return false;
    }
}

// Extract token from various sources
$token = null;
$authHeader = $_SERVER['HTTP_AUTHORIZATION'] ?? $_SERVER['HTTP_X_AUTHORIZATION'] ?? null;

if ($authHeader) {
    if (preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
        $token = $matches[1];
    } elseif (preg_match('/Token\s+(.*)$/i', $authHeader, $matches)) {
        $token = $matches[1];
    } else {
        $token = $authHeader;
    }
} elseif (isset($_GET['token'])) {
    $token = $_GET['token'];
} elseif (isset($_POST['token'])) {
    $token = $_POST['token'];
}

// Validate token
$user = validateToken($pdo, $token);
if (!$user) {
    http_response_code(401);
    echo json_encode(['error' => 'Invalid or expired token']);
    exit();
}

// Get request parameters
$action = $_GET['action'] ?? $_POST['action'] ?? null;
$entity = $_GET['entity'] ?? $_POST['entity'] ?? null;
$id = $_GET['id'] ?? $_POST['id'] ?? null;
$method = $_SERVER['REQUEST_METHOD'];

// Finance API Class
class FinanceAPI {
    private $pdo;
    private $user;
    
    public function __construct($pdo, $user) {
        $this->pdo = $pdo;
        $this->user = $user;
    }
    
    // Response helper
    private function response($data, $status = 200) {
        http_response_code($status);
        echo json_encode($data);
        exit();
    }
    
    // Error response helper
    private function error($message, $status = 400) {
        $this->response(['error' => $message], $status);
    }
    
    // Get current academic session
    private function getCurrentSession() {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM academic_sessions WHERE is_default = 1 LIMIT 1");
            $stmt->execute();
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            return null;
        }
    }
    
    // INCOME OPERATIONS
    public function getIncomes() {
        try {
            $page = $_GET['page'] ?? 1;
            $perPage = $_GET['per_page'] ?? 10;
            $offset = ($page - 1) * $perPage;
            
            $whereClause = "WHERE 1=1";
            $params = [];
            
            // Add filters
            if (isset($_GET['date_from'])) {
                $whereClause .= " AND date_of_expense >= ?";
                $params[] = $_GET['date_from'];
            }
            if (isset($_GET['date_to'])) {
                $whereClause .= " AND date_of_expense <= ?";
                $params[] = $_GET['date_to'];
            }
            if (isset($_GET['transaction_category_id'])) {
                $whereClause .= " AND transaction_category_id = ?";
                $params[] = $_GET['transaction_category_id'];
            }
            
            $sql = "SELECT i.*, tc.name as category_name, pm.name as payment_method_name 
                    FROM incomes i 
                    LEFT JOIN transaction_categories tc ON i.transaction_category_id = tc.id 
                    LEFT JOIN payment_methods pm ON i.payment_method_id = pm.id 
                    $whereClause ORDER BY i.date DESC, i.created_at DESC 
                    LIMIT $perPage OFFSET $offset";
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            $incomes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Get total count
            $countSql = "SELECT COUNT(*) FROM incomes i $whereClause";
            $countStmt = $this->pdo->prepare($countSql);
            $countStmt->execute($params);
            $total = $countStmt->fetchColumn();
            
            $this->response([
                'incomes' => $incomes,
                'pagination' => [
                    'current_page' => (int)$page,
                    'per_page' => (int)$perPage,
                    'total' => (int)$total,
                    'last_page' => ceil($total / $perPage)
                ]
            ]);
        } catch (Exception $e) {
            $this->error('Failed to fetch incomes: ' . $e->getMessage(), 500);
        }
    }
    
    public function getIncome($uuid) {
        try {
            $stmt = $this->pdo->prepare("SELECT i.*, tc.name as category_name, pm.name as payment_method_name 
                                        FROM incomes i 
                                        LEFT JOIN transaction_categories tc ON i.transaction_category_id = tc.id 
                                        LEFT JOIN payment_methods pm ON i.payment_method_id = pm.id 
                                        WHERE i.uuid = ?");
            $stmt->execute([$uuid]);
            $income = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$income) {
                $this->error('Income not found', 404);
            }
            
            $this->response(['income' => $income]);
        } catch (Exception $e) {
            $this->error('Failed to fetch income: ' . $e->getMessage(), 500);
        }
    }
    
    public function createIncome() {
        try {
            $data = json_decode(file_get_contents('php://input'), true) ?: $_POST;
            
            // Validate required fields
            $required = ['transaction_category_id', 'payment_method_id', 'amount', 'date'];
            foreach ($required as $field) {
                if (!isset($data[$field]) || empty($data[$field])) {
                    $this->error("Field '$field' is required");
                }
            }
            
            $uuid = uniqid('inc_', true);
            $stmt = $this->pdo->prepare("INSERT INTO incomes (uuid, transaction_category_id, payment_method_id, amount, date, description, created_at, updated_at) 
                                        VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW())");
            
            $stmt->execute([
                $uuid,
                $data['transaction_category_id'],
                $data['payment_method_id'],
                $data['amount'],
                $data['date'],
                $data['description'] ?? null
            ]);
            
            $this->response(['message' => 'Income created successfully', 'uuid' => $uuid], 201);
        } catch (Exception $e) {
            $this->error('Failed to create income: ' . $e->getMessage(), 500);
        }
    }
    
    // EXPENSE OPERATIONS
    public function getExpenses() {
        try {
            $page = $_GET['page'] ?? 1;
            $perPage = $_GET['per_page'] ?? 10;
            $offset = ($page - 1) * $perPage;
            
            $whereClause = "WHERE 1=1";
            $params = [];
            
            // Add filters
            if (isset($_GET['date_from'])) {
                $whereClause .= " AND date >= ?";
                $params[] = $_GET['date_from'];
            }
            if (isset($_GET['date_to'])) {
                $whereClause .= " AND date <= ?";
                $params[] = $_GET['date_to'];
            }
            if (isset($_GET['transaction_category_id'])) {
                $whereClause .= " AND transaction_category_id = ?";
                $params[] = $_GET['transaction_category_id'];
            }
            
            $sql = "SELECT e.*, tc.name as category_name 
                    FROM expenses e 
                    LEFT JOIN transaction_categories tc ON e.transaction_category_id = tc.id 
                    $whereClause ORDER BY e.date_of_expense DESC, e.created_at DESC 
                    LIMIT $perPage OFFSET $offset";
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            $expenses = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Get total count
            $countSql = "SELECT COUNT(*) FROM expenses e $whereClause";
            $countStmt = $this->pdo->prepare($countSql);
            $countStmt->execute($params);
            $total = $countStmt->fetchColumn();
            
            $this->response([
                'expenses' => $expenses,
                'pagination' => [
                    'current_page' => (int)$page,
                    'per_page' => (int)$perPage,
                    'total' => (int)$total,
                    'last_page' => ceil($total / $perPage)
                ]
            ]);
        } catch (Exception $e) {
            $this->error('Failed to fetch expenses: ' . $e->getMessage(), 500);
        }
    }
    
    public function getExpense($uuid) {
        try {
            $stmt = $this->pdo->prepare("SELECT e.*, tc.name as category_name 
                                        FROM expenses e 
                                        LEFT JOIN transaction_categories tc ON e.transaction_category_id = tc.id 
                                        WHERE e.uuid = ?");
            $stmt->execute([$uuid]);
            $expense = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$expense) {
                $this->error('Expense not found', 404);
            }
            
            $this->response(['expense' => $expense]);
        } catch (Exception $e) {
            $this->error('Failed to fetch expense: ' . $e->getMessage(), 500);
        }
    }
    
    public function createExpense() {
        try {
            $data = json_decode(file_get_contents('php://input'), true) ?: $_POST;
            
            // Validate required fields
            $required = ['transaction_category_id', 'amount', 'date'];
            foreach ($required as $field) {
                if (!isset($data[$field]) || empty($data[$field])) {
                    $this->error("Field '$field' is required");
                }
            }
            
            $uuid = uniqid('exp_', true);
            $stmt = $this->pdo->prepare("INSERT INTO expenses (uuid, transaction_category_id, amount, date_of_expense, description, created_at, updated_at) 
                                        VALUES (?, ?, ?, ?, ?, NOW(), NOW())");
            
            $stmt->execute([
                $uuid,
                $data['transaction_category_id'],
                $data['amount'],
                $data['date'],
                $data['description'] ?? null
            ]);
            
            $this->response(['message' => 'Expense created successfully', 'uuid' => $uuid], 201);
        } catch (Exception $e) {
            $this->error('Failed to create expense: ' . $e->getMessage(), 500);
        }
    }
    
    // ACCOUNT OPERATIONS
    public function getAccounts() {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM accounts ORDER BY name");
            $stmt->execute();
            $accounts = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $this->response(['accounts' => $accounts]);
        } catch (Exception $e) {
            $this->error('Failed to fetch accounts: ' . $e->getMessage(), 500);
        }
    }
    
    public function getAccount($id) {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM accounts WHERE id = ?");
            $stmt->execute([$id]);
            $account = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$account) {
                $this->error('Account not found', 404);
            }
            
            $this->response(['account' => $account]);
        } catch (Exception $e) {
            $this->error('Failed to fetch account: ' . $e->getMessage(), 500);
        }
    }
    
    // TRANSACTION CATEGORY OPERATIONS
    public function getTransactionCategories() {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM transaction_categories ORDER BY name");
            $stmt->execute();
            $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $this->response(['transaction_categories' => $categories]);
        } catch (Exception $e) {
            $this->error('Failed to fetch transaction categories: ' . $e->getMessage(), 500);
        }
    }
    
    // PAYMENT METHOD OPERATIONS
    public function getPaymentMethods() {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM payment_methods ORDER BY name");
            $stmt->execute();
            $methods = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $this->response(['payment_methods' => $methods]);
        } catch (Exception $e) {
            $this->error('Failed to fetch payment methods: ' . $e->getMessage(), 500);
        }
    }
    
    // DASHBOARD SUMMARY
    public function getDashboardSummary() {
        try {
            $session = $this->getCurrentSession();
            $sessionFilter = $session ? "AND created_at >= '{$session['starts']}' AND created_at <= '{$session['ends']}'" : "";
            
            // Total Income
            $stmt = $this->pdo->prepare("SELECT COALESCE(SUM(amount), 0) as total_income FROM incomes WHERE 1=1 $sessionFilter");
            $stmt->execute();
            $totalIncome = $stmt->fetchColumn();
            
            // Total Expense
            $stmt = $this->pdo->prepare("SELECT COALESCE(SUM(amount), 0) as total_expense FROM expenses WHERE 1=1 $sessionFilter");
            $stmt->execute();
            $totalExpense = $stmt->fetchColumn();
            
            // Recent Transactions
            $stmt = $this->pdo->prepare("(SELECT 'income' as type, uuid, amount, date, description, created_at FROM incomes ORDER BY created_at DESC LIMIT 5) 
                                        UNION ALL 
                                        (SELECT 'expense' as type, uuid, amount, date, description, created_at FROM expenses ORDER BY created_at DESC LIMIT 5) 
                                        ORDER BY created_at DESC LIMIT 10");
            $stmt->execute();
            $recentTransactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $this->response([
                'summary' => [
                    'total_income' => (float)$totalIncome,
                    'total_expense' => (float)$totalExpense,
                    'net_balance' => (float)($totalIncome - $totalExpense),
                    'current_session' => $session
                ],
                'recent_transactions' => $recentTransactions
            ]);
        } catch (Exception $e) {
            $this->error('Failed to fetch dashboard summary: ' . $e->getMessage(), 500);
        }
    }
    
    // PREREQUISITE DATA
    public function getPreRequisites() {
        try {
            $accounts = $this->pdo->query("SELECT * FROM accounts ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);
            $categories = $this->pdo->query("SELECT * FROM transaction_categories ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);
            $paymentMethods = $this->pdo->query("SELECT * FROM payment_methods ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);
            $session = $this->getCurrentSession();
            
            $this->response([
                'accounts' => $accounts,
                'transaction_categories' => $categories,
                'payment_methods' => $paymentMethods,
                'current_session' => $session
            ]);
        } catch (Exception $e) {
            $this->error('Failed to fetch prerequisites: ' . $e->getMessage(), 500);
        }
    }
}

// Initialize API
$api = new FinanceAPI($pdo, $user);

// Route requests
try {
    switch ($method) {
        case 'GET':
            switch ($entity) {
                case 'incomes':
                    if ($id) {
                        $api->getIncome($id);
                    } else {
                        $api->getIncomes();
                    }
                    break;
                    
                case 'expenses':
                    if ($id) {
                        $api->getExpense($id);
                    } else {
                        $api->getExpenses();
                    }
                    break;
                    
                case 'accounts':
                    if ($id) {
                        $api->getAccount($id);
                    } else {
                        $api->getAccounts();
                    }
                    break;
                    
                case 'transaction_categories':
                    $api->getTransactionCategories();
                    break;
                    
                case 'payment_methods':
                    $api->getPaymentMethods();
                    break;
                    
                case 'dashboard':
                    $api->getDashboardSummary();
                    break;
                    
                case 'prerequisites':
                    $api->getPreRequisites();
                    break;
                    
                default:
                    http_response_code(400);
                    echo json_encode(['error' => 'Invalid entity specified']);
            }
            break;
            
        case 'POST':
            switch ($entity) {
                case 'incomes':
                    $api->createIncome();
                    break;
                    
                case 'expenses':
                    $api->createExpense();
                    break;
                    
                default:
                    http_response_code(400);
                    echo json_encode(['error' => 'Invalid entity for POST request']);
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error: ' . $e->getMessage()]);
}

?>