<?php
error_log("URI: " . $_SERVER['REQUEST_URI']);
error_log("Parsed URI: " . print_r(explode('/', parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH)), true));
// Headers similar to course-batch.php
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include necessary files
require_once '../config/Database.php';
require_once '../middleware/Auth.php';

try {
    // Initialize database
    $database = new Database();
    $db = $database->getConnection();
    
    // Authentication - similar to course-batch.php
    $auth = new Auth($db);
    
    // Get token from headers or GET
    $token = null;
    $allHeaders = getallheaders();
    
    foreach ($allHeaders as $name => $value) {
        if (strtolower($name) === 'authorization') {
            if (preg_match('/Bearer\s+(.+)/', $value, $matches)) {
                $token = $matches[1];
            } else {
                $token = $value;
            }
            break;
        }
    }
    
    // Try other token sources if needed
    // (Same as in course-batch.php)
    
    // Validate token
    if (!$token) {
        http_response_code(401);
        echo json_encode(array(
            "status" => false,
            "message" => "No authorization token provided"
        ));
        exit();
    }
    
    $user = $auth->validateToken($token);
    
    if (!$user) {
        http_response_code(401);
        echo json_encode(array(
            "status" => false,
            "message" => "Invalid or expired token"
        ));
        exit();
    }
    
    // Exam Management Class
    class ExamManager {
        private $conn;
        
        public function __construct($db) {
            $this->conn = $db;
        }
        
        // Get all exams or a specific exam
        public function getExams($examId = null, $filters = array()) {
    try {
        $query = "SELECT DISTINCT
                    e.id, 
                    e.name, 
                    e.position,
                    e.description,
                    e.options,
                    e.created_at,
                    e.updated_at,
                    acs.id as academic_session_id,
                    acs.name as academic_session_name,
                    et.id as exam_term_id,
                    et.name as exam_term_name
                  FROM exams e
                  LEFT JOIN academic_sessions acs ON e.academic_session_id = acs.id
                  LEFT JOIN exam_terms et ON e.exam_term_id = et.id";
        
        // Join exam_schedules and batches if course or batch filters are present
        if (!empty($filters['course_id']) || !empty($filters['batch_id'])) {
            $query .= " LEFT JOIN exam_schedules es ON e.id = es.exam_id
                        LEFT JOIN batches b ON es.batch_id = b.id";
        }
        
        $query .= " WHERE 1=1";
        
        $params = array();
        
        // Add filters
        if (!empty($filters['academic_session_id'])) {
            $query .= " AND e.academic_session_id = :academic_session_id";
            $params[':academic_session_id'] = $filters['academic_session_id'];
        }
        
        if (!empty($filters['exam_term_id'])) {
            $query .= " AND e.exam_term_id = :exam_term_id";
            $params[':exam_term_id'] = $filters['exam_term_id'];
        }
        
        // Add course filter
        if (!empty($filters['course_id'])) {
            $query .= " AND b.course_id = :course_id";
            $params[':course_id'] = $filters['course_id'];
        }
        
        // Add batch filter
        if (!empty($filters['batch_id'])) {
            $query .= " AND es.batch_id = :batch_id";
            $params[':batch_id'] = $filters['batch_id'];
        }
        
        // Filter by specific exam ID if provided
        if ($examId) {
            $query .= " AND e.id = :exam_id";
            $params[':exam_id'] = $examId;
        }
        
        $query .= " ORDER BY e.position, e.name";
        
        $stmt = $this->conn->prepare($query);
        
        // Bind parameters
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        
        $stmt->execute();
        
        $exams = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // If a specific exam ID was requested
        if ($examId) {
            if (empty($exams)) {
                return null;
            }
            
            $exam = $exams[0];
            
            // Get related schedules
            $exam['schedules'] = $this->getExamSchedules($examId);
            
            // Get exam records for each schedule
            $exam['records'] = $this->getExamRecordsForExam($examId);
            
            return $exam;
        }
        
        return $exams;
    } catch(PDOException $e) {
        throw new Exception("Error fetching exams: " . $e->getMessage());
    }
}
        // Get exam records for a specific exam
public function getExamRecordsForExam($examId) {
    try {
        $query = "SELECT 
                    er.id,
                    er.uuid,
                    er.exam_schedule_id,
                    er.subject_id,
                    er.date,
                    er.start,
                    er.time,
                    er.options,
                    er.created_at,
                    er.updated_at,
                    s.name as subject_name,
                    s.code as subject_code,
                    es.batch_id,
                    b.name as batch_name,
                    c.id as course_id,
                    c.name as course_name
                  FROM exam_records er
                  JOIN exam_schedules es ON er.exam_schedule_id = es.id
                  JOIN subjects s ON er.subject_id = s.id
                  JOIN batches b ON es.batch_id = b.id
                  JOIN courses c ON b.course_id = c.id
                  WHERE es.exam_id = :exam_id
                  ORDER BY er.date, er.start, s.name";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':exam_id', $examId);
        $stmt->execute();
        
        $records = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Remove marks field to avoid exposing potentially sensitive data
        foreach ($records as &$record) {
            unset($record['marks']);
        }
        
        return $records;
    } catch(PDOException $e) {
        throw new Exception("Error fetching exam records: " . $e->getMessage());
    }
}
// Get marks for a specific exam record
public function getExamRecordMarks($examRecordId) {
    try {
        $query = "SELECT 
                    er.id,
                    er.marks,
                    er.subject_id,
                    s.name as subject_name,
                    s.code as subject_code,
                    es.batch_id,
                    b.name as batch_name,
                    e.id as exam_id,
                    e.name as exam_name
                  FROM exam_records er
                  JOIN exam_schedules es ON er.exam_schedule_id = es.id
                  JOIN exams e ON es.exam_id = e.id
                  JOIN subjects s ON er.subject_id = s.id
                  JOIN batches b ON es.batch_id = b.id
                  WHERE er.id = :exam_record_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':exam_record_id', $examRecordId);
        $stmt->execute();
        
        $record = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$record) {
            return null;
        }
        
        // Parse the marks JSON data
        if (isset($record['marks']) && !empty($record['marks'])) {
            $record['marks_data'] = json_decode($record['marks'], true);
        } else {
            $record['marks_data'] = [];
        }
        
        return $record;
    } catch(PDOException $e) {
        throw new Exception("Error fetching exam record marks: " . $e->getMessage());
    }
}
        // Get exam schedules for a specific exam
       public function getExamSchedules($examId) {
    try {
        $query = "SELECT 
                    es.id,
                    es.exam_id,
                    es.batch_id,
                    es.exam_grade_id,
                    es.exam_assessment_id,
                    es.observation_marks,
                    es.description,
                    es.options,
                    b.name as batch_name,
                    c.id as course_id,
                    c.name as course_name,
                    eg.name as grade_name,
                    ea.name as assessment_name
                  FROM exam_schedules es
                  LEFT JOIN batches b ON es.batch_id = b.id
                  LEFT JOIN courses c ON b.course_id = c.id
                  LEFT JOIN exam_grades eg ON es.exam_grade_id = eg.id
                  LEFT JOIN exam_assessments ea ON es.exam_assessment_id = ea.id
                  WHERE es.exam_id = :exam_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':exam_id', $examId);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch(PDOException $e) {
        throw new Exception("Error fetching exam schedules: " . $e->getMessage());
    }
}
        // Create a new exam
        public function createExam($data) {
            try {
                $query = "INSERT INTO exams 
                          (academic_session_id, exam_term_id, name, position, description, options) 
                          VALUES 
                          (:academic_session_id, :exam_term_id, :name, :position, :description, :options)";
                
                $stmt = $this->conn->prepare($query);
                
                // Sanitize and bind data
                $academicSessionId = $data->academic_session_id;
                $examTermId = $data->exam_term_id;
                $name = htmlspecialchars(strip_tags($data->name));
                $position = isset($data->position) ? $data->position : 0;
                $description = isset($data->description) ? htmlspecialchars(strip_tags($data->description)) : null;
                $options = isset($data->options) ? json_encode($data->options) : null;
                
                $stmt->bindParam(':academic_session_id', $academicSessionId);
                $stmt->bindParam(':exam_term_id', $examTermId);
                $stmt->bindParam(':name', $name);
                $stmt->bindParam(':position', $position);
                $stmt->bindParam(':description', $description);
                $stmt->bindParam(':options', $options);
                
                if ($stmt->execute()) {
                    return $this->conn->lastInsertId();
                }
                
                return false;
            } catch(PDOException $e) {
                throw new Exception("Error creating exam: " . $e->getMessage());
            }
        }
        
        // Update an exam
        public function updateExam($id, $data) {
            try {
                $query = "UPDATE exams 
                          SET 
                            academic_session_id = :academic_session_id,
                            exam_term_id = :exam_term_id,
                            name = :name,
                            position = :position,
                            description = :description,
                            options = :options,
                            updated_at = NOW()
                          WHERE id = :id";
                
                $stmt = $this->conn->prepare($query);
                
                // Sanitize and bind data
                $academicSessionId = $data->academic_session_id;
                $examTermId = $data->exam_term_id;
                $name = htmlspecialchars(strip_tags($data->name));
                $position = isset($data->position) ? $data->position : 0;
                $description = isset($data->description) ? htmlspecialchars(strip_tags($data->description)) : null;
                $options = isset($data->options) ? json_encode($data->options) : null;
                
                $stmt->bindParam(':academic_session_id', $academicSessionId);
                $stmt->bindParam(':exam_term_id', $examTermId);
                $stmt->bindParam(':name', $name);
                $stmt->bindParam(':position', $position);
                $stmt->bindParam(':description', $description);
                $stmt->bindParam(':options', $options);
                $stmt->bindParam(':id', $id);
                
                return $stmt->execute();
            } catch(PDOException $e) {
                throw new Exception("Error updating exam: " . $e->getMessage());
            }
        }
        
        // Delete an exam
        public function deleteExam($id) {
            try {
                // First check if there are any exam records linked to this exam
                $checkQuery = "SELECT COUNT(*) FROM exam_records er
                              JOIN exam_schedules es ON er.exam_schedule_id = es.id
                              WHERE es.exam_id = :id";
                              
                $checkStmt = $this->conn->prepare($checkQuery);
                $checkStmt->bindParam(':id', $id);
                $checkStmt->execute();
                
                if ($checkStmt->fetchColumn() > 0) {
                    throw new Exception("Cannot delete this exam because it has associated exam records.");
                }
                
                // Begin transaction
                $this->conn->beginTransaction();
                
                // Delete associated exam schedules first
                $deleteSchedulesQuery = "DELETE FROM exam_schedules WHERE exam_id = :id";
                $deleteSchedulesStmt = $this->conn->prepare($deleteSchedulesQuery);
                $deleteSchedulesStmt->bindParam(':id', $id);
                $deleteSchedulesStmt->execute();
                
                // Now delete the exam
                $deleteExamQuery = "DELETE FROM exams WHERE id = :id";
                $deleteExamStmt = $this->conn->prepare($deleteExamQuery);
                $deleteExamStmt->bindParam(':id', $id);
                $deleteExamStmt->execute();
                
                // Commit transaction
                $this->conn->commit();
                
                return true;
            } catch(PDOException $e) {
                // Rollback transaction on error
                $this->conn->rollBack();
                throw new Exception("Error deleting exam: " . $e->getMessage());
            }
        }
        
        // Get exam terms
        public function getExamTerms($academicSessionId = null) {
            try {
                $query = "SELECT 
                            et.id, 
                            et.name, 
                            et.position,
                            et.description,
                            et.options,
                            et.academic_session_id,
                            et.course_group_id,
                            acs.name as academic_session_name,
                            cg.name as course_group_name
                          FROM exam_terms et
                          LEFT JOIN academic_sessions acs ON et.academic_session_id = acs.id
                          LEFT JOIN course_groups cg ON et.course_group_id = cg.id";
                
                if ($academicSessionId) {
                    $query .= " WHERE et.academic_session_id = :academic_session_id";
                }
                
                $query .= " ORDER BY et.position, et.name";
                
                $stmt = $this->conn->prepare($query);
                
                if ($academicSessionId) {
                    $stmt->bindParam(':academic_session_id', $academicSessionId);
                }
                
                $stmt->execute();
                
                return $stmt->fetchAll(PDO::FETCH_ASSOC);
            } catch(PDOException $e) {
                throw new Exception("Error fetching exam terms: " . $e->getMessage());
            }
        }
        
        // Create an exam schedule
        public function createExamSchedule($data) {
            try {
                $query = "INSERT INTO exam_schedules 
                          (exam_id, batch_id, exam_grade_id, exam_assessment_id, observation_marks, description, options) 
                          VALUES 
                          (:exam_id, :batch_id, :exam_grade_id, :exam_assessment_id, :observation_marks, :description, :options)";
                
                $stmt = $this->conn->prepare($query);
                
                $examId = $data->exam_id;
                $batchId = $data->batch_id;
                $examGradeId = isset($data->exam_grade_id) ? $data->exam_grade_id : null;
                $examAssessmentId = isset($data->exam_assessment_id) ? $data->exam_assessment_id : null;
                $observationMarks = isset($data->observation_marks) ? json_encode($data->observation_marks) : null;
                $description = isset($data->description) ? htmlspecialchars(strip_tags($data->description)) : null;
                $options = isset($data->options) ? json_encode($data->options) : null;
                
                $stmt->bindParam(':exam_id', $examId);
                $stmt->bindParam(':batch_id', $batchId);
                $stmt->bindParam(':exam_grade_id', $examGradeId);
                $stmt->bindParam(':exam_assessment_id', $examAssessmentId);
                $stmt->bindParam(':observation_marks', $observationMarks);
                $stmt->bindParam(':description', $description);
                $stmt->bindParam(':options', $options);
                
                if ($stmt->execute()) {
                    return $this->conn->lastInsertId();
                }
                
                return false;
            } catch(PDOException $e) {
                throw new Exception("Error creating exam schedule: " . $e->getMessage());
            }
        }
        
        // Create an exam record
        public function createExamRecord($data) {
            try {
                $query = "INSERT INTO exam_records 
                          (exam_schedule_id, subject_id, date, start, time, marks, options) 
                          VALUES 
                          (:exam_schedule_id, :subject_id, :date, :start, :time, :marks, :options)";
                
                $stmt = $this->conn->prepare($query);
                
                $examScheduleId = $data->exam_schedule_id;
                $subjectId = $data->subject_id;
                $date = $data->date;
                $start = isset($data->start) ? $data->start : null;
                $time = isset($data->time) ? $data->time : null;
                $marks = isset($data->marks) ? json_encode($data->marks) : null;
                $options = isset($data->options) ? json_encode($data->options) : null;
                
                $stmt->bindParam(':exam_schedule_id', $examScheduleId);
                $stmt->bindParam(':subject_id', $subjectId);
                $stmt->bindParam(':date', $date);
                $stmt->bindParam(':start', $start);
                $stmt->bindParam(':time', $time);
                $stmt->bindParam(':marks', $marks);
                $stmt->bindParam(':options', $options);
                
                if ($stmt->execute()) {
                    return $this->conn->lastInsertId();
                }
                
                return false;
            } catch(PDOException $e) {
                throw new Exception("Error creating exam record: " . $e->getMessage());
            }
        }
        
        // Update marks for an exam record
        // Update marks for an exam record
public function updateExamMarks($id, $marks) {
    try {
        // Transform the marks data from the input format to the required database format
        $transformedMarks = [];
        
        // Convert the stdClass object to an array
        $marksArray = json_decode(json_encode($marks), true);
        
        foreach ($marksArray as $studentId => $markDetails) {
            $studentData = [
                "id" => $studentId,
                "is_absent" => isset($markDetails['is_absent']) ? $markDetails['is_absent'] : false,
                "assessment_details" => [
                    [
                        "id" => "5", // Assuming "5" is the default assessment ID
                        "ob" => $markDetails['obtained'],
                        "is_absent" => isset($markDetails['is_absent']) ? $markDetails['is_absent'] : false,
                        "comment" => null
                    ]
                ],
                "comment" => isset($markDetails['remarks']) ? $markDetails['remarks'] : null
            ];
            
            $transformedMarks[] = $studentData;
        }
        
        // Update the exam record with the transformed marks
        $query = "UPDATE exam_records SET marks = :marks, updated_at = NOW() WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        
        $marksJson = json_encode($transformedMarks);
        
        $stmt->bindParam(':marks', $marksJson);
        $stmt->bindParam(':id', $id);
        
        return $stmt->execute();
    } catch(PDOException $e) {
        throw new Exception("Error updating exam marks: " . $e->getMessage());
    }
}
       // Get student results
public function getStudentResults($studentId, $academicSessionId = null) {
    try {
        $query = "SELECT 
                    er.id as record_id,
                    er.marks,
                    er.date,
                    er.time,
                    s.id as subject_id,
                    s.name as subject_name,
                    e.id as exam_id,
                    e.name as exam_name,
                    et.name as exam_term_name,
                    b.id as batch_id,
                    b.name as batch_name,
                    sr.id as student_record_id
                  FROM exam_records er
                  JOIN exam_schedules es ON er.exam_schedule_id = es.id
                  JOIN exams e ON es.exam_id = e.id
                  JOIN exam_terms et ON e.exam_term_id = et.id
                  JOIN subjects s ON er.subject_id = s.id
                  JOIN batches b ON es.batch_id = b.id
                  JOIN student_records sr ON sr.batch_id = b.id
                  WHERE sr.student_id = :student_id";
        
        if ($academicSessionId) {
            $query .= " AND e.academic_session_id = :academic_session_id";
        }
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':student_id', $studentId);
        
        if ($academicSessionId) {
            $stmt->bindParam(':academic_session_id', $academicSessionId);
        }
        
        $stmt->execute();
        
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Process marks
        foreach ($results as &$result) {
            if (isset($result['marks']) && !empty($result['marks'])) {
                $marksData = json_decode($result['marks'], true);
                
                // Check if marksData is an array of student records
                if (is_array($marksData) && !isset($marksData[$studentId])) {
                    // Find the student's marks in the array
                    $studentMarks = null;
                    foreach ($marksData as $mark) {
                        if (isset($mark['id']) && $mark['id'] == $studentId) {
                            $studentMarks = $mark;
                            break;
                        }
                    }
                    $result['student_marks'] = $studentMarks;
                } else {
                    // Handle old format if it exists (with student IDs as keys)
                    $result['student_marks'] = isset($marksData[$studentId]) ? $marksData[$studentId] : null;
                }
            } else {
                $result['student_marks'] = null;
            }
            
            // Remove the raw marks data to avoid sending all students' data
            unset($result['marks']);
        }
        
        return $results;
    } catch(PDOException $e) {
        throw new Exception("Error fetching student results: " . $e->getMessage());
    }
}
}
    
    // Initialize ExamManager
    $examManager = new ExamManager($db);
    
    // Determine request method and route
    $method = $_SERVER['REQUEST_METHOD'];
    $uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
    $uri = explode('/', $uri);
    
    // Extract ID if present
    $id = null;
    if (isset($uri[4]) && is_numeric($uri[4])) {
        $id = intval($uri[4]);
    }
    
    // Handle request based on method and resource
    $response = array();
    
    
    
    // Handle GET /exams or GET /exams/{id}
    if ($method === 'GET' && $uri[3] === 'exams') {
    // Get filters if any
    $filters = array();
    if (isset($_GET['academic_session_id'])) {
        $filters['academic_session_id'] = intval($_GET['academic_session_id']);
    }
    if (isset($_GET['exam_term_id'])) {
        $filters['exam_term_id'] = intval($_GET['exam_term_id']);
    }
    if (isset($_GET['course_id'])) {
        $filters['course_id'] = intval($_GET['course_id']);
    }
    if (isset($_GET['batch_id'])) {
        $filters['batch_id'] = intval($_GET['batch_id']);
    }
    
    if ($id) {
        // Get specific exam
        $exam = $examManager->getExams($id, $filters);
        
        if ($exam) {
            $response = array(
                "status" => true,
                "message" => "Exam retrieved successfully",
                "data" => $exam
            );
            http_response_code(200);
        } else {
            $response = array(
                "status" => false,
                "message" => "Exam not found"
            );
            http_response_code(404);
        }
    } else {
        // Get all exams
        $exams = $examManager->getExams(null, $filters);
        
        $response = array(
            "status" => true,
            "message" => "Exams retrieved successfully",
            "data" => $exams
        );
        http_response_code(200);
    }
}
    // Handle POST /exams
    else if ($method === 'POST' && $uri[2] === 'exams' && !$id) {
        // Get posted data
        $data = json_decode(file_get_contents("php://input"));
        
        // Validate required fields
        if (!isset($data->academic_session_id) || !isset($data->exam_term_id) || !isset($data->name)) {
            $response = array(
                "status" => false,
                "message" => "Missing required fields: academic_session_id, exam_term_id, name"
            );
            http_response_code(400);
        } else {
            // Create exam
            $newExamId = $examManager->createExam($data);
            
            if ($newExamId) {
                $response = array(
                    "status" => true,
                    "message" => "Exam created successfully",
                    "data" => array(
                        "id" => $newExamId
                    )
                );
                http_response_code(201);
            } else {
                $response = array(
                    "status" => false,
                    "message" => "Failed to create exam"
                );
                http_response_code(500);
            }
        }
    }
    // Handle PUT /exams/{id}
    else if ($method === 'PUT' && $uri[2] === 'exams' && $id) {
        // Get posted data
        $data = json_decode(file_get_contents("php://input"));
        
        // Validate required fields
        if (!isset($data->academic_session_id) || !isset($data->exam_term_id) || !isset($data->name)) {
            $response = array(
                "status" => false,
                "message" => "Missing required fields: academic_session_id, exam_term_id, name"
            );
            http_response_code(400);
        } else {
            // Check if exam exists
            $exam = $examManager->getExams($id);
            
            if (!$exam) {
                $response = array(
                    "status" => false,
                    "message" => "Exam not found"
                );
                http_response_code(404);
            } else {
                // Update exam
                $result = $examManager->updateExam($id, $data);
                
                if ($result) {
                    $response = array(
                        "status" => true,
                        "message" => "Exam updated successfully"
                    );
                    http_response_code(200);
                } else {
                    $response = array(
                        "status" => false,
                        "message" => "Failed to update exam"
                    );
                    http_response_code(500);
                }
            }
        }
    }
    // Handle DELETE /exams/{id}
    else if ($method === 'DELETE' && $uri[2] === 'exams' && $id) {
        // Check if exam exists
        $exam = $examManager->getExams($id);
        
        if (!$exam) {
            $response = array(
                "status" => false,
                "message" => "Exam not found"
            );
            http_response_code(404);
        } else {
            // Try to delete exam
            try {
                $result = $examManager->deleteExam($id);
                
                $response = array(
                    "status" => true,
                    "message" => "Exam deleted successfully"
                );
                http_response_code(200);
            } catch (Exception $e) {
                $response = array(
                    "status" => false,
                    "message" => $e->getMessage()
                );
                http_response_code(400);
            }
        }
    }
    // Handle GET /exam-terms
    else if ($method === 'GET' && $uri[3] === 'exam-terms') {
        $academicSessionId = isset($_GET['academic_session_id']) ? intval($_GET['academic_session_id']) : null;
        
        $examTerms = $examManager->getExamTerms($academicSessionId);
        
        $response = array(
            "status" => true,
            "message" => "Exam terms retrieved successfully",
            "data" => $examTerms
        );
        http_response_code(200);
    }
    // Handle GET /exam-records/{id}/marks
else if ($method === 'GET' && $uri[3] === 'exam-records' && $id && isset($uri[5]) && $uri[5] === 'marks') {
    // Get exam record marks
    $marks = $examManager->getExamRecordMarks($id);
    
    if ($marks) {
        $response = array(
            "status" => true,
            "message" => "Exam record marks retrieved successfully",
            "data" => $marks
        );
        http_response_code(200);
    } else {
        $response = array(
            "status" => false,
            "message" => "Exam record not found"
        );
        http_response_code(404);
    }
}
    // Handle POST /exam-schedules
    else if ($method === 'POST' && $uri[3] === 'exam-schedules' && !$id) {
        // Get posted data
        $data = json_decode(file_get_contents("php://input"));
        
        // Validate required fields
        if (!isset($data->exam_id) || !isset($data->batch_id)) {
            $response = array(
                "status" => false,
                "message" => "Missing required fields: exam_id, batch_id"
            );
            http_response_code(400);
        } else {
            // Create exam schedule
            $newScheduleId = $examManager->createExamSchedule($data);
            
            if ($newScheduleId) {
                $response = array(
                    "status" => true,
                    "message" => "Exam schedule created successfully",
                    "data" => array(
                        "id" => $newScheduleId
                    )
                );
                http_response_code(201);
            } else {
                $response = array(
                    "status" => false,
                    "message" => "Failed to create exam schedule"
                );
                http_response_code(500);
            }
        }
    }
    // Handle POST /exam-records
    else if ($method === 'POST' && $uri[3] === 'exam-records' && !$id) {
        // Get posted data
        $data = json_decode(file_get_contents("php://input"));
        
        // Validate required fields
        if (!isset($data->exam_schedule_id) || !isset($data->subject_id) || !isset($data->date)) {
            $response = array(
                "status" => false,
                "message" => "Missing required fields: exam_schedule_id, subject_id, date"
            );
            http_response_code(400);
        } else {
            // Create exam record
            $newRecordId = $examManager->createExamRecord($data);
            
            if ($newRecordId) {
                $response = array(
                    "status" => true,
                    "message" => "Exam record created successfully",
                    "data" => array(
                        "id" => $newRecordId
                    )
                );
                http_response_code(201);
            } else {
                $response = array(
                    "status" => false,
                    "message" => "Failed to create exam record"
                );
                http_response_code(500);
            }
        }
    }
    // Handle POST /exam-records/{id}/marks
else if ($method === 'POST' && $uri[3] === 'exam-records' && $id && isset($uri[5]) && $uri[5] === 'marks') {
        // Get posted data
        $data = json_decode(file_get_contents("php://input"));
        // Validate marks data
    if (!isset($data->marks) || !is_object($data->marks)) {
        $response = array(
            "status" => false,
            "message" => "Invalid marks data format"
        );
        http_response_code(400);
    } else {
        // Update exam marks
        $result = $examManager->updateExamMarks($id, $data->marks);
        
        if ($result) {
            $response = array(
                "status" => true,
                "message" => "Exam marks updated successfully"
            );
            http_response_code(200);
        } else {
            $response = array(
                "status" => false,
                "message" => "Failed to update exam marks"
            );
            http_response_code(500);
        }
    }
}
// Handle GET /exam-results/student/{id}
else if ($method === 'GET' && $uri[3] === 'exam-results' && isset($uri[4]) && $uri[4] === 'student' && isset($uri[5]) && is_numeric($uri[5])) {
    $studentId = intval($uri[5]);
    $academicSessionId = isset($_GET['academic_session_id']) ? intval($_GET['academic_session_id']) : null;
    
    $results = $examManager->getStudentResults($studentId, $academicSessionId);
    
    $response = array(
        "status" => true,
        "message" => "Student results retrieved successfully",
        "data" => $results
    );
    http_response_code(200);
}
// If the requested resource or method is not found
else {
    $response = array(
        "status" => false,
        "message" => "Endpoint not found"
    );
    http_response_code(404);
}

// Return JSON response
echo json_encode($response);
} catch(Exception $e) {
http_response_code(500);
echo json_encode(array(
"status" => false,
"message" => "Server error occurred",
"error" => $e->getMessage()
));
}
?>
