<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Headers
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

try {
    // Include database and auth class
    require_once '../config/Database.php';
    require_once '../middleware/Auth.php';

    // Initialize database
    $database = new Database();
    $db = $database->getConnection();

    // Check database connection
    if (!$db) {
        throw new Exception("Database connection failed");
    }

    // Initialize Auth
    $auth = new Auth($db);

    // Get the token from the Authorization header
    $headers = getallheaders();
    $token = null;

    if (isset($headers['Authorization'])) {
        // Remove 'Bearer ' prefix if it exists
        $token = trim(str_replace('Bearer ', '', $headers['Authorization']));
    }

    // Validate token
    if (!$token) {
        http_response_code(401);
        echo json_encode(array(
            "status" => false,
            "message" => "No authorization token provided"
        ));
        exit();
    }

    $user = $auth->validateToken($token);
    if (!$user) {
        http_response_code(401);
        echo json_encode(array(
            "status" => false,
            "message" => "Invalid or expired token"
        ));
        exit();
    }

    class StudentDetails {
        private $conn;
        
        public function __construct($db) {
            $this->conn = $db;
        }
        
        public function getStudentsByBatch($batchId = null) {
            try {
                $query = "SELECT 
                            sr.id as student_record_id,
                            sr.roll_number,
                            sr.date_of_entry,
                            sr.batch_id,
                            sr.admission_id,
                            sr.student_id,
                            sr.total_fee,
                            sr.paid_fee,
                            sr.is_promoted,
                            sr.house_id,
                            s.uuid,
                            s.first_name,
                            s.middle_name,
                            s.last_name,
                            s.date_of_birth,
                            s.gender,
                            s.contact_number,
                            s.email,
                            s.blood_group_id,
                            s.religion_id,
                            s.category_id,
                            s.caste_id,
                            s.nationality,
                            s.present_address_line_1,
                            s.present_city,
                            s.present_state,
                            s.present_zipcode,
                            s.present_country,
                            s.permanent_address_line_1,
                            s.permanent_city,
                            s.permanent_state,
                            s.permanent_zipcode,
                            s.permanent_country,
                            s.mother_tongue,
                            s.unique_identification_number,
                            s.student_photo,
                            s.emergency_contact_name,
                            s.emergency_contact_number,
                            bg.name as blood_group,
                            rel.name as religion,
                            cat.name as category,
                            c.name as caste,
                            mi.name as medium_instruction,
                            h.name as house_name,
                            b.name as batch_name,
                            co.name as course_name,
                            co.id as course_id,
                            acs.name as academic_session_name,
                            acs.id as academic_session_id,
                            a.date_of_admission,
                            a.admission_remarks
                        FROM student_records sr
                        INNER JOIN students s ON sr.student_id = s.id
                        LEFT JOIN blood_groups bg ON s.blood_group_id = bg.id
                        LEFT JOIN religions rel ON s.religion_id = rel.id
                        LEFT JOIN categories cat ON s.category_id = cat.id
                        LEFT JOIN castes c ON s.caste_id = c.id
                        LEFT JOIN medium_instructions mi ON s.medium_instruction_id = mi.id
                        LEFT JOIN houses h ON sr.house_id = h.id
                        LEFT JOIN batches b ON sr.batch_id = b.id
                        LEFT JOIN courses co ON b.course_id = co.id
                        LEFT JOIN academic_sessions acs ON co.academic_session_id = acs.id
                        LEFT JOIN admissions a ON sr.admission_id = a.id";
                
                if($batchId) {
                    $query .= " WHERE sr.batch_id = :batch_id";
                }
                
                $stmt = $this->conn->prepare($query);
                
                if($batchId) {
                    $stmt->bindParam(":batch_id", $batchId, PDO::PARAM_INT);
                }
                
                $stmt->execute();
                
                $students = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if(empty($students)) {
                    return array();
                }
                
                // Format dates and process student data
                foreach($students as &$student) {
                    if(isset($student['date_of_birth'])) {
                        $student['date_of_birth'] = date('Y-m-d', strtotime($student['date_of_birth']));
                    }
                    if(isset($student['date_of_entry'])) {
                        $student['date_of_entry'] = date('Y-m-d', strtotime($student['date_of_entry']));
                    }
                    if(isset($student['date_of_admission'])) {
                        $student['date_of_admission'] = date('Y-m-d', strtotime($student['date_of_admission']));
                    }
                    
                    // Get parent information
                    $student['parent_info'] = $this->getParentInfo($student['student_id']);
                    
                    // Get fee information
                    $student['fee_records'] = $this->getStudentFeeRecords($student['student_record_id']);
                    
                    // Get attendance summary (if needed)
                    $student['attendance_summary'] = $this->getAttendanceSummary($student['student_record_id']);
                    
                    // Get academic performance (like exam records)
                    $student['academic_records'] = $this->getAcademicRecords($student['student_record_id']);
                    
                    // Get transport information (if applicable)
                    $student['transport_info'] = $this->getTransportInfo($student['student_record_id']);
                    
                    // Get documents
                    $student['documents'] = $this->getStudentDocuments($student['student_id']);
                }
                
                return $students;
                
            } catch(PDOException $e) {
                throw new Exception("Error fetching student details: " . $e->getMessage());
            }
        }
        
        public function getStudentsByCourse($courseId = null) {
            try {
                // First get all batches under this course
                $query = "SELECT id FROM batches WHERE course_id = :course_id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":course_id", $courseId, PDO::PARAM_INT);
                $stmt->execute();
                
                $batches = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if(empty($batches)) {
                    return array();
                }
                
                $allStudents = array();
                
                // For each batch, get the students
                foreach($batches as $batch) {
                    $batchStudents = $this->getStudentsByBatch($batch['id']);
                    $allStudents = array_merge($allStudents, $batchStudents);
                }
                
                return $allStudents;
                
            } catch(PDOException $e) {
                throw new Exception("Error fetching students by course: " . $e->getMessage());
            }
        }
        
        public function getParentInfo($studentId) {
            try {
                $query = "SELECT 
                            sp.id,
                            sp.first_guardian_name,
                            sp.first_guardian_relation,
                            sp.first_guardian_occupation,
                            sp.first_guardian_email,
                            sp.first_guardian_contact_number_1,
                            sp.first_guardian_contact_number_2,
                            sp.first_guardian_photo,
                            sp.second_guardian_name,
                            sp.second_guardian_relation,
                            sp.second_guardian_occupation,
                            sp.second_guardian_email,
                            sp.second_guardian_contact_number_1,
                            sp.second_guardian_contact_number_2,
                            sp.second_guardian_photo,
                            sp.third_guardian_name,
                            sp.third_guardian_relation
                        FROM students s
                        LEFT JOIN student_parents sp ON s.student_parent_id = sp.id
                        WHERE s.id = :student_id";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":student_id", $studentId, PDO::PARAM_INT);
                $stmt->execute();
                
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                
                return $result ?: array();
                
            } catch(PDOException $e) {
                return array(); // Return empty array on error
            }
        }
        
        public function getStudentFeeRecords($studentRecordId) {
            try {
                $query = "SELECT 
                            sfr.id,
                            sfr.fee_installment_id,
                            sfr.due_date,
                            sfr.status,
                            sfr.transport_fee,
                            sfr.late_fee,
                            sfr.late_fee_charged,
                            sfr.custom_fee,
                            fi.title as installment_title,
                            fc.name as fee_concession,
                            ft.name as transport_fee_name,
                            (SELECT SUM(sfrd.amount) FROM student_fee_record_details sfrd WHERE sfrd.student_fee_record_id = sfr.id) as total_amount,
                            (SELECT SUM(amount) FROM transactions WHERE student_fee_record_id = sfr.id AND is_cancelled = 0) as amount_paid
                        FROM student_fee_records sfr
                        LEFT JOIN fee_installments fi ON sfr.fee_installment_id = fi.id
                        LEFT JOIN fee_concessions fc ON sfr.fee_concession_id = fc.id
                        LEFT JOIN transport_fees ft ON sfr.transport_circle_id = ft.id
                        WHERE sfr.student_record_id = :student_record_id";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":student_record_id", $studentRecordId, PDO::PARAM_INT);
                $stmt->execute();
                
                $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if(!empty($result)) {
                    // Format dates
                    foreach($result as &$record) {
                        if(isset($record['due_date'])) {
                            $record['due_date'] = date('Y-m-d', strtotime($record['due_date']));
                        }
                    }
                }
                
                return $result ?: array();
                
            } catch(PDOException $e) {
                return array(); // Return empty array on error
            }
        }
        
        public function getAttendanceSummary($studentRecordId) {
    try {
        // Get batch id for this student record
        $query = "SELECT batch_id FROM student_records WHERE id = :student_record_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":student_record_id", $studentRecordId, PDO::PARAM_INT);
        $stmt->execute();
        
        $batchId = $stmt->fetchColumn();
        
        if(!$batchId) {
            return array();
        }
        
        // Get attendance records and calculate summary
        $query = "SELECT 
                    date_of_attendance,
                    attendance
                FROM student_attendances 
                WHERE batch_id = :batch_id
                ORDER BY date_of_attendance DESC
                LIMIT 30"; // Get last 30 days of attendance
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":batch_id", $batchId, PDO::PARAM_INT);
        $stmt->execute();
        
        $attendanceRecords = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if(empty($attendanceRecords)) {
            return array();
        }
        
        $summary = array(
            'total_days' => count($attendanceRecords),
            'present' => 0,
            'absent' => 0,
            'leave' => 0,
            'late' => 0,
            'half_day' => 0,
            'attendance_percentage' => 0,
            'recent_records' => array()
        );
        
        foreach($attendanceRecords as $record) {
            // Attendance is stored as JSON
            $attendanceData = json_decode($record['attendance'], true);
            
            if(!is_array($attendanceData)) {
                continue;
            }
            
            $status = 'present'; // Default status
            
            // Check if student is in the 'data' array (absent)
            if(isset($attendanceData['data']) && is_array($attendanceData['data'])) {
                foreach($attendanceData['data'] as $absentStudent) {
                    if(isset($absentStudent['id']) && $absentStudent['id'] == $studentRecordId) {
                        $status = 'absent';
                        break;
                    }
                }
            }
            
            // Check if student is in the 'late' array
            if($status == 'present' && isset($attendanceData['late']) && is_array($attendanceData['late'])) {
                foreach($attendanceData['late'] as $lateStudent) {
                    if(isset($lateStudent['id']) && $lateStudent['id'] == $studentRecordId) {
                        $status = 'late';
                        break;
                    }
                }
            }
            
            // Check if student is in the 'half_day' array
            if($status == 'present' && isset($attendanceData['half_day']) && is_array($attendanceData['half_day'])) {
                foreach($attendanceData['half_day'] as $halfDayStudent) {
                    if(isset($halfDayStudent['id']) && $halfDayStudent['id'] == $studentRecordId) {
                        $status = 'half_day';
                        break;
                    }
                }
            }
            
            // Update summary counts
            if($status == 'present') {
                $summary['present']++;
            } else if($status == 'absent') {
                $summary['absent']++;
            } else if($status == 'late') {
                $summary['late']++;
                $summary['present']++; // Count late as present too for attendance percentage
            } else if($status == 'half_day') {
                $summary['half_day']++;
                $summary['present'] += 0.5; // Count half day as 0.5 present for attendance percentage
            }
            
            // Add to recent records (limited to 5)
            if(count($summary['recent_records']) < 5) {
                $summary['recent_records'][] = array(
                    'date' => date('Y-m-d', strtotime($record['date_of_attendance'])),
                    'status' => $status
                );
            }
        }
        
        // Round present count to integer to avoid type conflicts in the app
        $summary['present'] = (int)$summary['present'];
        
        // Calculate attendance percentage with limited decimal precision (1 decimal place)
        if($summary['total_days'] > 0) {
            $percentage = ($summary['present'] / $summary['total_days']) * 100;
            $summary['attendance_percentage'] = round($percentage, 1); // Round to 1 decimal place
        } else {
            $summary['attendance_percentage'] = 0.0; // Ensure it's always a float
        }
        
        return $summary;
        
    } catch(PDOException $e) {
        // Return a properly formatted empty summary with consistent types
        return array(
            'total_days' => 0,
            'present' => 0,
            'absent' => 0,
            'leave' => 0,
            'late' => 0,
            'half_day' => 0,
            'attendance_percentage' => 0.0, // Ensure it's a float
            'recent_records' => array()
        );
    }
}
        
        public function getAcademicRecords($studentRecordId) {
    try {
        // Get exam records for this student
        $query = "SELECT 
                    er.id,
                    er.uuid,
                    er.exam_schedule_id,
                    er.subject_id,
                    er.date,
                    s.name as subject_name,
                    e.name as exam_name,
                    e.id as exam_id,
                    et.name as term_name,
                    er.marks
                FROM exam_records er
                LEFT JOIN exam_schedules es ON er.exam_schedule_id = es.id
                LEFT JOIN exams e ON es.exam_id = e.id
                LEFT JOIN exam_terms et ON e.exam_term_id = et.id
                LEFT JOIN subjects s ON er.subject_id = s.id
                LEFT JOIN batches b ON es.batch_id = b.id
                LEFT JOIN student_records sr ON sr.batch_id = b.id
                WHERE sr.id = :student_record_id
                ORDER BY er.date DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":student_record_id", $studentRecordId, PDO::PARAM_INT);
        $stmt->execute();
        
        $examRecords = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $academicRecords = array(
            'exams' => array(),
            'summary' => array(
                'total_exams' => 0,
                'total_subjects' => 0,
                'top_subject' => '',
                'highest_score' => 0
            )
        );
        
        if(!empty($examRecords)) {
            $subjectScores = array();
            $examCount = 0;
            $highestScore = 0;
            $topSubject = '';
            
            foreach($examRecords as $record) {
                // Process the marks JSON data to extract this student's marks
                $marksData = json_decode($record['marks'], true);
                $studentMarks = null;
                
                if(is_array($marksData)) {
                    // Check if data is an array of objects with 'id' field
                    foreach($marksData as $mark) {
                        if(isset($mark['id']) && $mark['id'] == $studentRecordId) {
                            $studentMarks = $mark;
                            break;
                        }
                    }
                    
                    // If not found, check if data is keyed by student IDs
                    if(!$studentMarks && isset($marksData[$studentRecordId])) {
                        $studentMarks = $marksData[$studentRecordId];
                    }
                }
                
                if($studentMarks) {
                    $examRecord = array(
                        'exam_id' => $record['exam_id'],
                        'exam_name' => $record['exam_name'],
                        'term_name' => $record['term_name'],
                        'subject_id' => $record['subject_id'],
                        'subject_name' => $record['subject_name'],
                        'date' => $record['date'] ? date('Y-m-d', strtotime($record['date'])) : null,
                        'marks_obtained' => 0,
                        'max_marks' => 0,
                        'percentage' => 0
                    );
                    
                    // Extract marks from assessment details if available
                    if(isset($studentMarks['assessment_details']) && is_array($studentMarks['assessment_details'])) {
                        $totalObtained = 0;
                        $totalMax = 0;
                        
                        foreach($studentMarks['assessment_details'] as $assessment) {
                            if(isset($assessment['ob']) && !isset($assessment['is_absent']) || 
                               (isset($assessment['is_absent']) && $assessment['is_absent'] === false)) {
                                $totalObtained += floatval($assessment['ob']);
                                // Assuming max_mark exists or fallback to 100
                                $totalMax += isset($assessment['max_mark']) ? floatval($assessment['max_mark']) : 100;
                            }
                        }
                        
                        $examRecord['marks_obtained'] = $totalObtained;
                        $examRecord['max_marks'] = $totalMax;
                    } else {
                        // Fallback to direct marks if available
                        $examRecord['marks_obtained'] = isset($studentMarks['marks']) ? floatval($studentMarks['marks']) : 0;
                        $examRecord['max_marks'] = isset($studentMarks['max_mark']) ? floatval($studentMarks['max_mark']) : 100;
                    }
                    
                    // Calculate percentage
                    if($examRecord['max_marks'] > 0) {
                        $examRecord['percentage'] = round(($examRecord['marks_obtained'] / $examRecord['max_marks']) * 100, 2);
                    }
                    
                    $academicRecords['exams'][] = $examRecord;
                    $examCount++;
                    
                    // Track subject scores for finding top subject
                    $subjectId = $record['subject_id'];
                    if(!isset($subjectScores[$subjectId])) {
                        $subjectScores[$subjectId] = array(
                            'subject_name' => $record['subject_name'],
                            'total_score' => 0,
                            'count' => 0
                        );
                    }
                    
                    $subjectScores[$subjectId]['total_score'] += $examRecord['percentage'];
                    $subjectScores[$subjectId]['count']++;
                    
                    // Check if this is the highest score
                    if($examRecord['percentage'] > $highestScore) {
                        $highestScore = $examRecord['percentage'];
                        $topSubject = $record['subject_name'];
                    }
                }
            }
            
            // Update summary
            $academicRecords['summary'] = array(
                'total_exams' => $examCount,
                'total_subjects' => count($subjectScores),
                'top_subject' => $topSubject,
                'highest_score' => $highestScore
            );
        }
        
        return $academicRecords;
        
    } catch(PDOException $e) {
        return array('exams' => array(), 'summary' => array(
            'total_exams' => 0,
            'total_subjects' => 0,
            'top_subject' => '',
            'highest_score' => 0
        )); // Return empty arrays on error
    }
}
        
        public function getTransportInfo($studentRecordId) {
            try {
                $query = "SELECT 
                            sfr.transport_circle_id,
                            tc.name as transport_circle_name,
                            ts.name as transport_stop_name,
                            tr.name as transport_route_name,
                            tv.name as transport_vehicle_name,
                            tv.registration_number as vehicle_registration,
                            tv.capacity as vehicle_capacity
                        FROM student_fee_records sfr
                        LEFT JOIN transport_circles tc ON sfr.transport_circle_id = tc.id
                        LEFT JOIN transport_stops ts ON tc.transport_stop_id = ts.id
                        LEFT JOIN transport_routes tr ON ts.transport_route_id = tr.id
                        LEFT JOIN transport_vehicles tv ON tr.transport_vehicle_id = tv.id
                        WHERE sfr.student_record_id = :student_record_id
                        AND sfr.transport_circle_id IS NOT NULL
                        LIMIT 1";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":student_record_id", $studentRecordId, PDO::PARAM_INT);
                $stmt->execute();
                
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                
                return $result ?: array();
                
            } catch(PDOException $e) {
                return array(); // Return empty array on error
            }
        }
        
        public function getStudentDocuments($studentId) {
            try {
                $query = "SELECT 
                            sd.id,
                            sd.title,
                            sd.description,
                            sdt.name as document_type,
                            u.filename,
                            u.user_filename
                        FROM student_documents sd
                        LEFT JOIN student_document_types sdt ON sd.student_document_type_id = sdt.id
                        LEFT JOIN uploads u ON sd.upload_token = u.upload_token
                        WHERE sd.student_id = :student_id";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":student_id", $studentId, PDO::PARAM_INT);
                $stmt->execute();
                
                $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                return $result ?: array();
                
            } catch(PDOException $e) {
                return array(); // Return empty array on error
            }
        }
        
        public function getStudentById($studentId) {
            try {
                $query = "SELECT 
                            sr.id as student_record_id,
                            sr.roll_number,
                            sr.date_of_entry,
                            sr.batch_id,
                            sr.admission_id,
                            sr.student_id,
                            sr.total_fee,
                            sr.paid_fee,
                            sr.is_promoted,
                            sr.house_id,
                            s.uuid,
                            s.first_name,
                            s.middle_name,
                            s.last_name,
                            s.date_of_birth,
                            s.gender,
                            s.contact_number,
                            s.email,
                            s.blood_group_id,
                            s.religion_id,
                            s.category_id,
                            s.caste_id,
                            s.nationality,
                            s.present_address_line_1,
                            s.present_city,
                            s.present_state,
                            s.present_zipcode,
                            s.present_country,
                            s.permanent_address_line_1,
                            s.permanent_city,
                            s.permanent_state,
                            s.permanent_zipcode,
                            s.permanent_country,
                            s.mother_tongue,
                            s.unique_identification_number,
                            s.student_photo,
                            s.emergency_contact_name,
                            s.emergency_contact_number,
                            bg.name as blood_group,
                            rel.name as religion,
                            cat.name as category,
                            c.name as caste,
                            mi.name as medium_instruction,
                            h.name as house_name,
                            b.name as batch_name,
                            co.name as course_name,
                            co.id as course_id,
                            acs.name as academic_session_name,
                            acs.id as academic_session_id,
                            a.date_of_admission,
                            a.admission_remarks
                        FROM students s
                        LEFT JOIN student_records sr ON s.id = sr.student_id
                        LEFT JOIN blood_groups bg ON s.blood_group_id = bg.id
                        LEFT JOIN religions rel ON s.religion_id = rel.id
                        LEFT JOIN categories cat ON s.category_id = cat.id
                        LEFT JOIN castes c ON s.caste_id = c.id
                        LEFT JOIN medium_instructions mi ON s.medium_instruction_id = mi.id
                        LEFT JOIN houses h ON sr.house_id = h.id
                        LEFT JOIN batches b ON sr.batch_id = b.id
                        LEFT JOIN courses co ON b.course_id = co.id
                        LEFT JOIN academic_sessions acs ON co.academic_session_id = acs.id
                        LEFT JOIN admissions a ON sr.admission_id = a.id
                        WHERE s.id = :student_id";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":student_id", $studentId, PDO::PARAM_INT);
                $stmt->execute();
                
                $student = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if(!$student) {
                    return array();
                }
                
                // Format dates
                if(isset($student['date_of_birth'])) {
                    $student['date_of_birth'] = date('Y-m-d', strtotime($student['date_of_birth']));
                }
                if(isset($student['date_of_entry'])) {
                    $student['date_of_entry'] = date('Y-m-d', strtotime($student['date_of_entry']));
                }
                if(isset($student['date_of_admission'])) {
                    $student['date_of_admission'] = date('Y-m-d', strtotime($student['date_of_admission']));
                }
                
                // Get parent information
                $student['parent_info'] = $this->getParentInfo($student['student_id']);
                
                // Get fee information
                $student['fee_records'] = $this->getStudentFeeRecords($student['student_record_id']);
                
                // Get attendance summary
                $student['attendance_summary'] = $this->getAttendanceSummary($student['student_record_id']);
                
                // Get academic performance
                $student['academic_records'] = $this->getAcademicRecords($student['student_record_id']);
                
                // Get transport information
                $student['transport_info'] = $this->getTransportInfo($student['student_record_id']);
                
                // Get documents
                $student['documents'] = $this->getStudentDocuments($student['student_id']);
                
                return $student;
                
            } catch(PDOException $e) {
                throw new Exception("Error fetching student details: " . $e->getMessage());
            }
        }
    }

    // Initialize StudentDetails object
    $studentDetails = new StudentDetails($db);

    // Get request parameters
    $courseId = isset($_GET['course_id']) ? intval($_GET['course_id']) : null;
    $batchId = isset($_GET['batch_id']) ? intval($_GET['batch_id']) : null;
    $studentId = isset($_GET['student_id']) ? intval($_GET['student_id']) : null;

    // Get data based on parameters
    $response = array();

    if($studentId) {
        // Get a specific student
        $studentData = $studentDetails->getStudentById($studentId);
        
        if(empty($studentData)) {
            http_response_code(404);
            echo json_encode(array(
                "status" => false,
                "message" => "Student not found"
            ));
            exit();
        }
        
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "Student details retrieved successfully",
            "data" => $studentData
        ));
    } else if($batchId) {
        // Get students by batch
        $students = $studentDetails->getStudentsByBatch($batchId);
        
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "Student details retrieved successfully",
            "count" => count($students),
            "data" => $students
        ));
    } else if($courseId) {
        // Get students by course
        $students = $studentDetails->getStudentsByCourse($courseId);
        
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "Student details retrieved successfully",
            "count" => count($students),
            "data" => $students
        ));
    } else {
        // No specific parameters, return error
        http_response_code(400);
        echo json_encode(array(
            "status" => false,
            "message" => "Please provide at least one of: student_id, batch_id, or course_id"
        ));
    }

} catch(Exception $e) {
    http_response_code(500);
    echo json_encode(array(
        "status" => false,
        "message" => "Server error occurred",
        "error" => $e->getMessage()
    ));
}
?>