// Dashboard JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // Check Authentication
    if (!Auth.isAuthenticated()) {
        window.location.href = 'index.html';
        return;
    }
    
    // Elements
    const sidebar = document.querySelector('.sidebar');
    const sidebarToggle = document.getElementById('sidebarToggle');
    const menuToggle = document.getElementById('menuToggle');
    const userDropdown = document.querySelector('.user-dropdown');
    const logoutBtn = document.getElementById('logoutBtn');
    const logoutItem = document.querySelector('.logout-item');
    const loaderOverlay = document.querySelector('.loader-overlay');
    
    // Show loader
    loaderOverlay.classList.add('show');
    
    // Setup User Info
    setupUserInfo();
    
    // Fetch Dashboard Data
    fetchDashboardData();
    
    // Event Listeners
    sidebarToggle.addEventListener('click', toggleSidebar);
    menuToggle.addEventListener('click', toggleSidebarMobile);
    userDropdown.addEventListener('click', toggleDropdown);
    logoutBtn.addEventListener('click', logout);
    logoutItem.addEventListener('click', logout);
    
    // Setup Events outside dropdowns to close them
    document.addEventListener('click', function(e) {
        if (!userDropdown.contains(e.target)) {
            userDropdown.classList.remove('active');
        }
    });
    
    // Setup current date
    const currentDateElement = document.getElementById('currentDate');
    const today = new Date();
    const options = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
    currentDateElement.textContent = 'Today is ' + today.toLocaleDateString(undefined, options);
    
    // Initialize Charts
    initializeCharts();
    
    // Functions
    function toggleSidebar() {
        sidebar.classList.toggle('collapsed');
    }
    
    function toggleSidebarMobile() {
        sidebar.classList.toggle('active');
    }
    
    function toggleDropdown(e) {
        e.stopPropagation();
        userDropdown.classList.toggle('active');
    }
    
    function logout() {
        // Show loader
        loaderOverlay.classList.add('show');
        
        // Simulate API call
        setTimeout(() => {
            Auth.logout();
        }, 1000);
    }
    
    function setupUserInfo() {
        const userData = Auth.getUserData();
        
        if (userData) {
            // Set user name
            const userNameElements = document.querySelectorAll('#userName, #welcomeUserName');
            userNameElements.forEach(el => {
                el.textContent = userData.username || 'User';
            });
            
            // Set user role
            const userRoleElements = document.querySelectorAll('#userRole');
            userRoleElements.forEach(el => {
                el.textContent = userData.roles && userData.roles.length > 0 ? 
                    userData.roles[0].charAt(0).toUpperCase() + userData.roles[0].slice(1) : 
                    'User';
            });
            
            // Set user initials
            const initials = Utils.getInitials(userData.username || 'User');
            const userInitialsElements = document.querySelectorAll('#userInitials, #userAvatarSmall');
            userInitialsElements.forEach(el => {
                el.textContent = initials;
            });
        }
    }
    
    async function fetchDashboardData() {
        try {
            const response = await fetch('https://edmapi.ed5.in/api/dashboard.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': Auth.getToken()
                }
            });
            
            const data = await response.json();
            
            if (data.status) {
                // Populate dashboard with data
                populateStats(data.data.stats);
                populateActivity(data.data.activities);
                populateTasks(data.data.tasks);
                populateEvents(data.data.events);
                updateCharts(data.data.charts);
            } else {
                showNotification('Failed to load dashboard data', 'error');
            }
        } catch (error) {
            console.error('Dashboard data error:', error);
            showNotification('Error loading dashboard data', 'error');
        } finally {
            // Hide loader
            loaderOverlay.classList.remove('show');
        }
    }
    
    function populateStats(stats) {
        const statsContainer = document.getElementById('statsCards');
        
        // Clear skeleton loading
        statsContainer.innerHTML = '';
        
        // Check if stats exist
        if (!stats || stats.length === 0) {
            // Create sample stats if none available
            stats = [
                { title: 'Total Students', value: '1,250', icon: 'fa-user-graduate', type: 'students' },
                { title: 'Total Teachers', value: '85', icon: 'fa-chalkboard-teacher', type: 'teachers' },
                { title: 'Active Courses', value: '42', icon: 'fa-book', type: 'courses' },
                { title: 'Total Revenue', value: '₹ 1.5M', icon: 'fa-money-bill-wave', type: 'revenue' }
            ];
        }
        
        // Populate stats
        stats.forEach(stat => {
            const statCard = document.createElement('div');
            statCard.className = `stats-card ${stat.type || ''}`;
            statCard.innerHTML = `
                <div class="stats-icon">
                    <i class="fas ${stat.icon}"></i>
                </div>
                <div class="stats-info">
                    <p class="stats-title">${stat.title}</p>
                    <h3 class="stats-value">${stat.value}</h3>
                </div>
            `;
            statsContainer.appendChild(statCard);
        });
    }
    
    function populateActivity(activities) {
        const activityList = document.getElementById('activityList');
        
        // Clear skeleton loading
        activityList.innerHTML = '';
        
        // Check if activities exist
        if (!activities || activities.length === 0) {
            // Create sample activities if none available
            activities = [
                { 
                    title: 'New Student Enrolled', 
                    detail: 'Rahul Sharma was added to Class X-B', 
                    time: '2 hours ago',
                    type: 'new-student',
                    icon: 'fa-user-plus'
                },
                { 
                    title: 'Attendance Marked', 
                    detail: 'Class IX-A attendance recorded by Ms. Patel', 
                    time: '3 hours ago',
                    type: 'attendance',
                    icon: 'fa-clipboard-check'
                },
                { 
                    title: 'Fee Payment Received', 
                    detail: 'Anjali Singh paid ₹15,000 for Term 2', 
                    time: '5 hours ago',
                    type: 'fee-payment',
                    icon: 'fa-rupee-sign'
                },
                { 
                    title: 'Exam Schedule Updated', 
                    detail: 'Mid-term exams rescheduled to next week', 
                    time: '1 day ago',
                    type: 'exam',
                    icon: 'fa-clipboard-list'
                }
            ];
        }
        
        // Populate activities
        activities.forEach(activity => {
            const activityItem = document.createElement('li');
            activityItem.className = `activity-item ${activity.type || ''}`;
            activityItem.innerHTML = `
                <div class="activity-icon">
                    <i class="fas ${activity.icon}"></i>
                </div>
                <div class="activity-content">
                    <h4 class="activity-title">${activity.title}</h4>
                    <p class="activity-detail">${activity.detail}</p>
                </div>
                <div class="activity-time">${activity.time}</div>
            `;
            activityList.appendChild(activityItem);
        });
    }
    
    function populateTasks(tasks) {
        const taskList = document.getElementById('taskList');
        
        // Clear skeleton loading
        taskList.innerHTML = '';
        
        // Check if tasks exist
        if (!tasks || tasks.length === 0) {
            // Create sample tasks if none available
            tasks = [
                { title: 'Review new admission applications', date: 'Today', completed: false },
                { title: 'Prepare monthly attendance report', date: 'Tomorrow', completed: false },
                { title: 'Meeting with parents committee', date: 'March 15, 2025', completed: false },
                { title: 'Update science lab inventory', date: 'March 18, 2025', completed: true }
            ];
        }
        
        // Populate tasks
        tasks.forEach(task => {
            const taskItem = document.createElement('li');
            taskItem.className = 'task-item';
            taskItem.innerHTML = `
                <div class="task-checkbox ${task.completed ? 'checked' : ''}"></div>
                <div class="task-content">
                    <h4 class="task-title">${task.title}</h4>
                    <p class="task-date">${task.date}</p>
                </div>
            `;
            
            // Add event listener to checkbox
            taskItem.querySelector('.task-checkbox').addEventListener('click', function() {
                this.classList.toggle('checked');
            });
            
            taskList.appendChild(taskItem);
        });
    }
    
    function populateEvents(events) {
        const eventList = document.getElementById('eventList');
        
        // Clear skeleton loading
        eventList.innerHTML = '';
        
        // Check if events exist
        if (!events || events.length === 0) {
            // Create sample events if none available
            events = [
                { 
                    title: 'Annual Sports Day', 
                    details: 'School playground, 9:00 AM onwards', 
                    date: { month: 'Mar', day: '20' } 
                },
                { 
                    title: 'Parent-Teacher Meeting', 
                    details: 'All classrooms, 10:00 AM to 1:00 PM', 
                    date: { month: 'Mar', day: '25' } 
                },
                { 
                    title: 'Science Exhibition', 
                    details: 'School auditorium, All day event', 
                    date: { month: 'Apr', day: '05' } 
                },
                { 
                    title: 'Career Counseling Session', 
                    details: 'For Classes XI and XII, Seminar Hall', 
                    date: { month: 'Apr', day: '12' } 
                }
            ];
        }
        
        // Populate events
        events.forEach(event => {
            const eventItem = document.createElement('li');
            eventItem.className = 'event-item';
            eventItem.innerHTML = `
                <div class="event-date">
                    <span class="month">${event.date.month}</span>
                    <span class="day">${event.date.day}</span>
                </div>
                <div class="event-content">
                    <h4 class="event-title">${event.title}</h4>
                    <p class="event-details">${event.details}</p>
                </div>
            `;
            eventList.appendChild(eventItem);
        });
    }
    
    function initializeCharts() {
        // Enrollment Chart
        const enrollmentCtx = document.getElementById('enrollmentChart').getContext('2d');
        window.enrollmentChart = new Chart(enrollmentCtx, {
            type: 'bar',
            data: {
                labels: ['Class I', 'Class II', 'Class III', 'Class IV', 'Class V', 'Class VI', 'Class VII', 'Class VIII', 'Class IX', 'Class X'],
                datasets: [{
                    label: 'Students',
                    data: [45, 55, 60, 48, 52, 58, 50, 45, 42, 40],
                    backgroundColor: 'rgba(108, 99, 255, 0.7)',
                    borderColor: '#6c63ff',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0, 0, 0, 0.05)'
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });
        
        // Fee Collection Chart
        const feeCtx = document.getElementById('feeChart').getContext('2d');
        window.feeChart = new Chart(feeCtx, {
            type: 'line',
            data: {
                labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun'],
                datasets: [{
                    label: 'Fee Collection (in ₹)',
                    data: [250000, 300000, 350000, 280000, 320000, 400000],
                    backgroundColor: 'rgba(255, 99, 132, 0.2)',
                    borderColor: '#ff6384',
                    borderWidth: 2,
                    tension: 0.3,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0, 0, 0, 0.05)'
                        },
                        ticks: {
                            callback: function(value) {
                                return '₹' + value.toLocaleString();
                            }
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });
    }
    
    function updateCharts(chartData) {
        // Check if chart data exists
        if (!chartData) return;
        
        // Update enrollment chart if it exists
        if (chartData.enrollment && window.enrollmentChart) {
            window.enrollmentChart.data.labels = chartData.enrollment.labels || window.enrollmentChart.data.labels;
            window.enrollmentChart.data.datasets[0].data = chartData.enrollment.data || window.enrollmentChart.data.datasets[0].data;
            window.enrollmentChart.update();
        }
        
        // Update fee chart if it exists
        if (chartData.fees && window.feeChart) {
            window.feeChart.data.labels = chartData.fees.labels || window.feeChart.data.labels;
            window.feeChart.data.datasets[0].data = chartData.fees.data || window.feeChart.data.datasets[0].data;
            window.feeChart.update();
        }
    }
    
    // Notification function
    function showNotification(message, type = 'success') {
        const notification = document.querySelector('.notification');
        const notificationIcon = notification.querySelector('.notification-icon i');
        const notificationMessage = notification.querySelector('.notification-message');
        
        // Set message
        notificationMessage.textContent = message;
        
        // Set icon based on type
        if (type === 'success') {
            notificationIcon.className = 'fas fa-check-circle';
            notificationIcon.style.color = '#4CAF50';
        } else if (type === 'error') {
            notificationIcon.className = 'fas fa-times-circle';
            notificationIcon.style.color = '#F44336';
        } else if (type === 'warning') {
            notificationIcon.className = 'fas fa-exclamation-triangle';
            notificationIcon.style.color = '#FF9800';
        } else if (type === 'info') {
            notificationIcon.className = 'fas fa-info-circle';
            notificationIcon.style.color = '#2196F3';
        }
        
        // Show notification
        notification.classList.add('show');
        
        // Hide after 3 seconds
        setTimeout(() => {
            notification.classList.remove('show');
        }, 3000);
    }
});