<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Headers
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Debug: Log the raw request
error_log("Raw request received: " . file_get_contents('php://input'));

try {
    // Include database and auth class
    require_once '../config/Database.php';
    require_once '../middleware/Auth.php';

    // Initialize database
    $database = new Database();
    $db = $database->getConnection();

    // Check database connection
    if (!$db) {
        throw new Exception("Database connection failed");
    }

    // Initialize Auth
    $auth = new Auth($db);

    // Get the token from multiple possible sources
    $token = null;
    
    // 1. Try from getallheaders() first (case-insensitive)
    $allHeaders = getallheaders();
    error_log("All headers: " . print_r($allHeaders, true));
    
    foreach ($allHeaders as $name => $value) {
        if (strtolower($name) === 'authorization') {
            // Extract token - handle both "Bearer token" and raw token formats
            if (preg_match('/Bearer\s+(.+)/', $value, $matches)) {
                $token = $matches[1];
            } else {
                $token = $value;
            }
            error_log("Found token in getallheaders(): " . substr($token, 0, 10) . "...");
            break;
        }
    }
    
    // 2. Try from $_SERVER if still not found
    if (!$token) {
        // Apache specific
        if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
            $authHeader = $_SERVER['HTTP_AUTHORIZATION'];
            
            if (preg_match('/Bearer\s+(.+)/', $authHeader, $matches)) {
                $token = $matches[1];
            } else {
                $token = $authHeader;
            }
            error_log("Found token in \$_SERVER['HTTP_AUTHORIZATION']: " . substr($token, 0, 10) . "...");
        }
        // Apache with mod_rewrite
        elseif (isset($_SERVER['REDIRECT_HTTP_AUTHORIZATION'])) {
            $authHeader = $_SERVER['REDIRECT_HTTP_AUTHORIZATION'];
            
            if (preg_match('/Bearer\s+(.+)/', $authHeader, $matches)) {
                $token = $matches[1];
            } else {
                $token = $authHeader;
            }
            error_log("Found token in \$_SERVER['REDIRECT_HTTP_AUTHORIZATION']: " . substr($token, 0, 10) . "...");
        }
        // Try to get raw headers
        else {
            foreach ($_SERVER as $key => $value) {
                if (substr($key, 0, 5) === 'HTTP_') {
                    $key = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($key, 5)))));
                    if (strtolower($key) === 'authorization') {
                        if (preg_match('/Bearer\s+(.+)/', $value, $matches)) {
                            $token = $matches[1];
                        } else {
                            $token = $value;
                        }
                        error_log("Found token in \$_SERVER HTTP_ variable: " . substr($token, 0, 10) . "...");
                        break;
                    }
                }
            }
        }
    }
    
    // 3. Last resort: Look in GET parameters
    if (!$token && isset($_GET['token'])) {
        $token = $_GET['token'];
        error_log("Found token in GET parameter: " . substr($token, 0, 10) . "...");
    }

    // Validate token
    if (!$token) {
        http_response_code(401);
        echo json_encode(array(
            "status" => false,
            "message" => "No authorization token provided",
            "headers_received" => $allHeaders
        ));
        exit();
    }

    // Check if token exists in database before validation
    if (!$auth->tokenExists($token)) {
        error_log("Token does not exist in database: " . substr($token, 0, 10) . "...");
        http_response_code(401);
        echo json_encode(array(
            "status" => false,
            "message" => "Invalid token: not found in database"
        ));
        exit();
    }

    $user = $auth->validateToken($token);
    
    if (!$user) {
        http_response_code(401);
        echo json_encode(array(
            "status" => false,
            "message" => "Invalid or expired token"
        ));
        exit();
    }

    class StudentAttendance {
        private $conn;
        private $table_name = "student_attendances";
        
        public function __construct($db) {
            $this->conn = $db;
        }
        
        /**
         * Get students for a batch
         */
        public function getStudentsForBatch($batchId) {
            try {
                $query = "SELECT 
                            sr.id, 
                            s.first_name, 
                            s.middle_name,
                            s.last_name,
                            sr.roll_number,
                            s.uuid,
                            s.unique_identification_number,
                            s.gender,
                            s.contact_number,
                            s.student_photo
                        FROM student_records sr
                        JOIN students s ON sr.student_id = s.id
                        WHERE sr.batch_id = :batch_id
                        ORDER BY sr.roll_number ASC, s.first_name ASC";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":batch_id", $batchId, PDO::PARAM_INT);
                $stmt->execute();
                
                return $stmt->fetchAll(PDO::FETCH_ASSOC);
            } catch(PDOException $e) {
                throw new Exception("Error fetching students for batch: " . $e->getMessage());
            }
        }
        
        /**
         * Get attendance for a specific date and batch
         */
        public function getAttendanceByDate($batchId, $date, $subjectId = null) {
            try {
                $query = "SELECT * FROM " . $this->table_name . " 
                          WHERE batch_id = :batch_id 
                          AND date_of_attendance = :date";
                
                if ($subjectId) {
                    $query .= " AND subject_id = :subject_id";
                }
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":batch_id", $batchId, PDO::PARAM_INT);
                $stmt->bindParam(":date", $date);
                
                if ($subjectId) {
                    $stmt->bindParam(":subject_id", $subjectId, PDO::PARAM_INT);
                }
                
                $stmt->execute();
                
                $attendance = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($attendance) {
                    // Parse JSON attendance data
                    $attendance['attendance'] = json_decode($attendance['attendance'], true);
                    $attendance['options'] = $attendance['options'] ? json_decode($attendance['options'], true) : null;
                }
                
                return $attendance;
            } catch(PDOException $e) {
                throw new Exception("Error getting attendance: " . $e->getMessage());
            }
        }
        
        /**
         * Mark attendance for a batch on a specific date
         */
        public function markAttendance($batchId, $date, $attendanceData, $subjectId = null, $session = null, $options = null) {
            try {
                // First check if attendance record already exists for this batch and date
                $existingAttendance = $this->getAttendanceByDate($batchId, $date, $subjectId);
                
                if ($existingAttendance) {
                    // Update existing record
                    $query = "UPDATE " . $this->table_name . " SET 
                              attendance = :attendance,
                              options = :options,
                              updated_at = NOW()
                              WHERE id = :id";
                    
                    $stmt = $this->conn->prepare($query);
                    $stmt->bindParam(":id", $existingAttendance['id'], PDO::PARAM_INT);
                    
                    $attendanceJson = json_encode($attendanceData);
                    $optionsJson = json_encode($options);
                    
                    $stmt->bindParam(":attendance", $attendanceJson);
                    $stmt->bindParam(":options", $optionsJson);
                    
                    $result = $stmt->execute();
                    
                    return array(
                        "result" => $result,
                        "operation" => "update",
                        "id" => $existingAttendance['id']
                    );
                } else {
                    // Create new record
                    $query = "INSERT INTO " . $this->table_name . " 
                              (batch_id, subject_id, session, date_of_attendance, attendance, options, created_at, updated_at) 
                              VALUES 
                              (:batch_id, :subject_id, :session, :date, :attendance, :options, NOW(), NOW())";
                    
                    $stmt = $this->conn->prepare($query);
                    $stmt->bindParam(":batch_id", $batchId, PDO::PARAM_INT);
                    $stmt->bindParam(":date", $date);
                    
                    // Handle null/optional parameters
                    if ($subjectId) {
                        $stmt->bindParam(":subject_id", $subjectId, PDO::PARAM_INT);
                    } else {
                        $stmt->bindValue(":subject_id", null, PDO::PARAM_NULL);
                    }
                    
                    if ($session) {
                        $stmt->bindParam(":session", $session);
                    } else {
                        $stmt->bindValue(":session", null, PDO::PARAM_NULL);
                    }
                    
                    $attendanceJson = json_encode($attendanceData);
                    $optionsJson = json_encode($options);
                    
                    $stmt->bindParam(":attendance", $attendanceJson);
                    $stmt->bindParam(":options", $optionsJson);
                    
                    $result = $stmt->execute();
                    $newId = $this->conn->lastInsertId();
                    
                    return array(
                        "result" => $result,
                        "operation" => "create",
                        "id" => $newId
                    );
                }
            } catch(PDOException $e) {
                throw new Exception("Error marking attendance: " . $e->getMessage());
            }
        }
        
        /**
         * Get subjects for a batch
         */
        public function getSubjectsForBatch($batchId) {
            try {
                $query = "SELECT 
                            id, 
                            name, 
                            code, 
                            description
                        FROM subjects
                        WHERE batch_id = :batch_id
                        ORDER BY name ASC";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":batch_id", $batchId, PDO::PARAM_INT);
                $stmt->execute();
                
                return $stmt->fetchAll(PDO::FETCH_ASSOC);
            } catch(PDOException $e) {
                throw new Exception("Error fetching subjects for batch: " . $e->getMessage());
            }
        }
        
        /**
         * Get batch details
         */
        public function getBatchDetails($batchId) {
            try {
                $query = "SELECT 
                            b.id, 
                            b.name, 
                            b.description,
                            c.id as course_id,
                            c.name as course_name
                        FROM batches b
                        LEFT JOIN courses c ON b.course_id = c.id
                        WHERE b.id = :batch_id";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":batch_id", $batchId, PDO::PARAM_INT);
                $stmt->execute();
                
                return $stmt->fetch(PDO::FETCH_ASSOC);
            } catch(PDOException $e) {
                throw new Exception("Error fetching batch details: " . $e->getMessage());
            }
        }
        
        /**
         * Get attendance history for a batch
         */
        public function getAttendanceHistory($batchId, $fromDate = null, $toDate = null, $subjectId = null) {
            try {
                $query = "SELECT 
                            id, 
                            date_of_attendance, 
                            subject_id, 
                            session,
                            created_at,
                            updated_at
                        FROM " . $this->table_name . "
                        WHERE batch_id = :batch_id";
                
                if ($fromDate) {
                    $query .= " AND date_of_attendance >= :from_date";
                }
                
                if ($toDate) {
                    $query .= " AND date_of_attendance <= :to_date";
                }
                
                if ($subjectId) {
                    $query .= " AND subject_id = :subject_id";
                }
                
                $query .= " ORDER BY date_of_attendance DESC";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":batch_id", $batchId, PDO::PARAM_INT);
                
                if ($fromDate) {
                    $stmt->bindParam(":from_date", $fromDate);
                }
                
                if ($toDate) {
                    $stmt->bindParam(":to_date", $toDate);
                }
                
                if ($subjectId) {
                    $stmt->bindParam(":subject_id", $subjectId, PDO::PARAM_INT);
                }
                
                $stmt->execute();
                
                return $stmt->fetchAll(PDO::FETCH_ASSOC);
            } catch(PDOException $e) {
                throw new Exception("Error fetching attendance history: " . $e->getMessage());
            }
        }
        
        /**
         * Get attendance details
         */
        public function getAttendanceDetails($attendanceId) {
            try {
                $query = "SELECT * FROM " . $this->table_name . " WHERE id = :id";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":id", $attendanceId, PDO::PARAM_INT);
                $stmt->execute();
                
                $attendance = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($attendance) {
                    // Parse JSON attendance data
                    $attendance['attendance'] = json_decode($attendance['attendance'], true);
                    $attendance['options'] = $attendance['options'] ? json_decode($attendance['options'], true) : null;
                }
                
                return $attendance;
            } catch(PDOException $e) {
                throw new Exception("Error getting attendance details: " . $e->getMessage());
            }
        }
        
        /**
         * Delete attendance record
         */
        public function deleteAttendance($attendanceId) {
            try {
                $query = "DELETE FROM " . $this->table_name . " WHERE id = :id";
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":id", $attendanceId, PDO::PARAM_INT);
                
                return $stmt->execute();
            } catch(PDOException $e) {
                throw new Exception("Error deleting attendance: " . $e->getMessage());
            }
        }
        
        /**
         * Get student attendance statistics
         */
        public function getStudentAttendanceStats($batchId, $fromDate = null, $toDate = null, $studentId = null) {
            try {
                // Get all attendance records for this batch in date range
                $query = "SELECT id, date_of_attendance, attendance FROM " . $this->table_name . "
                          WHERE batch_id = :batch_id";
                
                if ($fromDate) {
                    $query .= " AND date_of_attendance >= :from_date";
                }
                
                if ($toDate) {
                    $query .= " AND date_of_attendance <= :to_date";
                }
                
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":batch_id", $batchId, PDO::PARAM_INT);
                
                if ($fromDate) {
                    $stmt->bindParam(":from_date", $fromDate);
                }
                
                if ($toDate) {
                    $stmt->bindParam(":to_date", $toDate);
                }
                
                $stmt->execute();
                $attendanceRecords = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                // Get students for this batch
                $students = $this->getStudentsForBatch($batchId);
                
                // Initialize statistics array
                $stats = array();
                foreach ($students as $student) {
                    if ($studentId && $student['id'] != $studentId) {
                        continue;
                    }
                    
                    $stats[$student['id']] = array(
                        'student_id' => $student['id'],
                        'name' => $student['first_name'] . ' ' . $student['last_name'],
                        'roll_number' => $student['roll_number'],
                        'admission_number' => $student['admission_number'],
                        'total_days' => count($attendanceRecords),
                        'present' => 0,
                        'absent' => 0,
                        'late' => 0,
                        'half_day' => 0,
                        'attendance_percentage' => 0
                    );
                }
                
                // Process attendance records
                foreach ($attendanceRecords as $record) {
                    $attendance = json_decode($record['attendance'], true);
                    
                    // Get list of absent students
                    $absentIds = array();
                    if (isset($attendance['data'])) {
                        foreach ($attendance['data'] as $absentStudent) {
                            $absentIds[] = $absentStudent['id'];
                        }
                    }
                    
                    // Get list of late students
                    $lateIds = array();
                    if (isset($attendance['late'])) {
                        foreach ($attendance['late'] as $lateStudent) {
                            $lateIds[] = $lateStudent['id'];
                        }
                    }
                    
                    // Get list of half day students
                    $halfDayIds = array();
                    if (isset($attendance['half_day'])) {
                        foreach ($attendance['half_day'] as $halfDayStudent) {
                            $halfDayIds[] = $halfDayStudent['id'];
                        }
                    }
                    
                    // Update statistics for each student
                    foreach ($stats as $studentId => &$studentStats) {
                        if (in_array($studentId, $absentIds)) {
                            $studentStats['absent']++;
                        } elseif (in_array($studentId, $lateIds)) {
                            $studentStats['late']++;
                            $studentStats['present']++; // Late is still counted as present
                        } elseif (in_array($studentId, $halfDayIds)) {
                            $studentStats['half_day']++;
                            $studentStats['present'] += 0.5; // Half day is counted as 0.5 present
                        } else {
                            $studentStats['present']++;
                        }
                    }
                }
                
                // Calculate percentages
                foreach ($stats as &$studentStats) {
                    if ($studentStats['total_days'] > 0) {
                        $studentStats['attendance_percentage'] = round(($studentStats['present'] / $studentStats['total_days']) * 100, 2);
                    }
                }
                
                return array_values($stats);
            } catch(PDOException $e) {
                throw new Exception("Error getting attendance statistics: " . $e->getMessage());
            }
        }
    }

    // Initialize StudentAttendance object
    $studentAttendance = new StudentAttendance($db);

    // Handle different HTTP methods
    $method = $_SERVER['REQUEST_METHOD'];
    
    switch ($method) {
        case 'GET':
            // Get request parameters
            $batchId = isset($_GET['batch_id']) ? intval($_GET['batch_id']) : null;
            $date = isset($_GET['date']) ? $_GET['date'] : null;
            $subjectId = isset($_GET['subject_id']) ? intval($_GET['subject_id']) : null;
            $attendanceId = isset($_GET['attendance_id']) ? intval($_GET['attendance_id']) : null;
            $action = isset($_GET['action']) ? $_GET['action'] : null;
            $fromDate = isset($_GET['from_date']) ? $_GET['from_date'] : null;
            $toDate = isset($_GET['to_date']) ? $_GET['to_date'] : null;
            $studentId = isset($_GET['student_id']) ? intval($_GET['student_id']) : null;
            
            $response = array();
            
            // Validate required parameters based on action
            if (!$action) {
                throw new Exception("Action parameter is required");
            }
            
            switch ($action) {
                case 'get_students':
                    if (!$batchId) {
                        throw new Exception("Batch ID is required for action: get_students");
                    }
                    
                    $students = $studentAttendance->getStudentsForBatch($batchId);
                    $response = array(
                        "status" => true,
                        "message" => "Students retrieved successfully",
                        "data" => array(
                            "batch_id" => $batchId,
                            "students" => $students
                        )
                    );
                    break;
                    
                case 'get_attendance':
                    if (!$batchId || !$date) {
                        throw new Exception("Batch ID and date are required for action: get_attendance");
                    }
                    
                    $attendance = $studentAttendance->getAttendanceByDate($batchId, $date, $subjectId);
                    
                    if (!$attendance) {
                        $response = array(
                            "status" => true,
                            "message" => "No attendance record found for the specified date",
                            "data" => null
                        );
                    } else {
                        $response = array(
                            "status" => true,
                            "message" => "Attendance retrieved successfully",
                            "data" => $attendance
                        );
                    }
                    break;
                    
                case 'get_attendance_details':
                    if (!$attendanceId) {
                        throw new Exception("Attendance ID is required for action: get_attendance_details");
                    }
                    
                    $attendance = $studentAttendance->getAttendanceDetails($attendanceId);
                    
                    if (!$attendance) {
                        $response = array(
                            "status" => false,
                            "message" => "Attendance record not found",
                            "data" => null
                        );
                    } else {
                        $response = array(
                            "status" => true,
                            "message" => "Attendance details retrieved successfully",
                            "data" => $attendance
                        );
                    }
                    break;
                    
                case 'get_attendance_history':
                    if (!$batchId) {
                        throw new Exception("Batch ID is required for action: get_attendance_history");
                    }
                    
                    $history = $studentAttendance->getAttendanceHistory($batchId, $fromDate, $toDate, $subjectId);
                    $response = array(
                        "status" => true,
                        "message" => "Attendance history retrieved successfully",
                        "data" => array(
                            "batch_id" => $batchId,
                            "history" => $history
                        )
                    );
                    break;
                    
                case 'get_subjects':
                    if (!$batchId) {
                        throw new Exception("Batch ID is required for action: get_subjects");
                    }
                    
                    $subjects = $studentAttendance->getSubjectsForBatch($batchId);
                    $response = array(
                        "status" => true,
                        "message" => "Subjects retrieved successfully",
                        "data" => array(
                            "batch_id" => $batchId,
                            "subjects" => $subjects
                        )
                    );
                    break;
                    
                case 'get_batch_details':
                    if (!$batchId) {
                        throw new Exception("Batch ID is required for action: get_batch_details");
                    }
                    
                    $batchDetails = $studentAttendance->getBatchDetails($batchId);
                    $response = array(
                        "status" => true,
                        "message" => "Batch details retrieved successfully",
                        "data" => $batchDetails
                    );
                    break;
                    
                case 'get_statistics':
                    if (!$batchId) {
                        throw new Exception("Batch ID is required for action: get_statistics");
                    }
                    
                    $stats = $studentAttendance->getStudentAttendanceStats($batchId, $fromDate, $toDate, $studentId);
                    $response = array(
                        "status" => true,
                        "message" => "Attendance statistics retrieved successfully",
                        "data" => array(
                            "batch_id" => $batchId,
                            "from_date" => $fromDate,
                            "to_date" => $toDate,
                            "statistics" => $stats
                        )
                    );
                    break;
                    
                default:
                    throw new Exception("Invalid action specified");
            }
            
            http_response_code(200);
            echo json_encode($response);
            break;
            
        case 'POST':
            // Get request data
            $data = json_decode(file_get_contents("php://input"), true);
            
            // Validate required parameters
            if (!isset($data['batch_id']) || !isset($data['date']) || !isset($data['attendance'])) {
                http_response_code(400);
                echo json_encode(array(
                    "status" => false,
                    "message" => "Incomplete data. Batch ID, date, and attendance data are required."
                ));
                exit();
            }
            
            $batchId = intval($data['batch_id']);
            $date = $data['date'];
            $attendanceData = $data['attendance'];
            $subjectId = isset($data['subject_id']) ? intval($data['subject_id']) : null;
            $session = isset($data['session']) ? $data['session'] : null;
            $options = isset($data['options']) ? $data['options'] : null;
            
            // Mark attendance
            $result = $studentAttendance->markAttendance($batchId, $date, $attendanceData, $subjectId, $session, $options);
            
            if ($result['result']) {
                http_response_code(200);
                echo json_encode(array(
                    "status" => true,
                    "message" => "Attendance " . ($result['operation'] === 'create' ? 'recorded' : 'updated') . " successfully",
                    "data" => array(
                        "id" => $result['id'],
                        "operation" => $result['operation']
                    )
                ));
            } else {
                http_response_code(500);
                echo json_encode(array(
                    "status" => false,
                    "message" => "Failed to " . ($result['operation'] === 'create' ? 'record' : 'update') . " attendance"
                ));
            }
            break;
            
        case 'PUT':
            // Get request data
            $data = json_decode(file_get_contents("php://input"), true);
            
            // Validate required parameters
            if (!isset($data['id']) || !isset($data['attendance'])) {
                http_response_code(400);
                echo json_encode(array(
                    "status" => false,
                    "message" => "Incomplete data. Attendance ID and attendance data are required."
                ));
                exit();
            }
            
            $attendanceId = intval($data['id']);
            $attendanceData = $data['attendance'];
            $options = isset($data['options']) ? $data['options'] : null;
            
            // Get existing attendance record
            $existingAttendance = $studentAttendance->getAttendanceDetails($attendanceId);
            
            if (!$existingAttendance) {
                http_response_code(404);
                echo json_encode(array(
                    "status" => false,
                    "message" => "Attendance record not found"
                ));
                exit();
            }
            
            // Update attendance
            $result = $studentAttendance->markAttendance(
                $existingAttendance['batch_id'],
                $existingAttendance['date_of_attendance'],
                $attendanceData,
                $existingAttendance['subject_id'],
                $existingAttendance['session'],
                $options
            );
            
            if ($result['result']) {
                http_response_code(200);
                echo json_encode(array(
                    "status" => true,
                    "message" => "Attendance updated successfully",
                    "data" => array(
                        "id" => $result['id']
                    )
                ));
            } else {
                http_response_code(500);
                echo json_encode(array(
                    "status" => false,
                    "message" => "Failed to update attendance"
                ));
            }
            break;
            
        case 'DELETE':
            // Get request parameters
            $attendanceId = isset($_GET['attendance_id']) ? intval($_GET['attendance_id']) : null;
            
            if (!$attendanceId) {
                http_response_code(400);
                echo json_encode(array(
                    "status" => false,
                    "message" => "Attendance ID is required"
                ));
                exit();
            }
            
            // Delete attendance record
            $result = $studentAttendance->deleteAttendance($attendanceId);
            
            if ($result) {
                http_response_code(200);
                echo json_encode(array(
                    "status" => true,
                    "message" => "Attendance record deleted successfully"
                ));
            } else {
                http_response_code(500);
                echo json_encode(array(
                    "status" => false,
                    "message" => "Failed to delete attendance record"
                ));
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(array(
                "status" => false,
                "message" => "Method not allowed"
            ));
            break;
    }

} catch(Exception $e) {
    http_response_code(500);
    echo json_encode(array(
        "status" => false,
        "message" => "Server error occurred",
        "error" => $e->getMessage(),
        "trace" => debug_backtrace()
    ));
}