// Login Page JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // Elements
    const loginForm = document.getElementById('loginForm');
    const usernameInput = document.getElementById('username');
    const passwordInput = document.getElementById('password');
    const loginBtn = document.querySelector('.login-btn');
    const togglePasswordBtn = document.getElementById('togglePassword');
    const loaderOverlay = document.querySelector('.loader-overlay');
    const notification = document.querySelector('.notification');
    
    // Apply animations on load
    animateLoginForm();
    
    // Add ripple effect to buttons
    addRippleEffect();
    
    // Password visibility toggle
    togglePasswordBtn.addEventListener('click', function() {
        if (passwordInput.type === 'password') {
            passwordInput.type = 'text';
            togglePasswordBtn.classList.remove('fa-eye');
            togglePasswordBtn.classList.add('fa-eye-slash');
        } else {
            passwordInput.type = 'password';
            togglePasswordBtn.classList.remove('fa-eye-slash');
            togglePasswordBtn.classList.add('fa-eye');
        }
    });
    
    // Form submit event
    loginForm.addEventListener('submit', function(e) {
        e.preventDefault();
        
        // Set button to loading state
        loginBtn.classList.add('loading');
        
        // Get form data
        const username = usernameInput.value;
        const password = passwordInput.value;
        
        // Create form data object
        const data = {
            username: username,
            password: password
        };
        
        // Call API
        loginUser(data);
    });
    
    // Login function
    async function loginUser(credentials) {
        try {
            const response = await fetch('https://edmapi.ed5.in/api/login.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(credentials)
            });
            
            const data = await response.json();
            
            // Reset button state
            loginBtn.classList.remove('loading');
            
            if (data.status === true) {
                // Store token in localStorage
                localStorage.setItem('auth_token', data.data.token);
                localStorage.setItem('user_data', JSON.stringify(data.data.user));
                
                // Show full page loader
                loaderOverlay.classList.add('show');
                
                // Show success notification
                showNotification('Login successful! Redirecting...', 'success');
                
                // Redirect after delay
                setTimeout(function() {
                    window.location.href = 'dashboard.html';
                }, 2000);
            } else {
                // Show error notification
                showNotification('Invalid username or password', 'error');
            }
        } catch (error) {
            console.error('Login error:', error);
            loginBtn.classList.remove('loading');
            showNotification('Server error, please try again later', 'error');
        }
    }
    
    // Animation for login form
    function animateLoginForm() {
        const loginContainer = document.querySelector('.login-container');
        const logoContainer = document.querySelector('.logo-container');
        const appTitle = document.querySelector('.app-title');
        const appSubtitle = document.querySelector('.app-subtitle');
        const formElements = document.querySelectorAll('.input-group, .form-options, .login-btn');
        
        // Add initial opacity 0
        logoContainer.style.opacity = '0';
        appTitle.style.opacity = '0';
        appSubtitle.style.opacity = '0';
        formElements.forEach(el => {
            el.style.opacity = '0';
        });
        
        // Container animation
        loginContainer.classList.add('animate-fadeIn');
        
        // Logo animation
        setTimeout(() => {
            logoContainer.style.opacity = '1';
            logoContainer.classList.add('animate-fadeIn');
        }, 300);
        
        // Titles animation
        setTimeout(() => {
            appTitle.style.opacity = '1';
            appTitle.classList.add('animate-fadeIn');
        }, 500);
        
        setTimeout(() => {
            appSubtitle.style.opacity = '1';
            appSubtitle.classList.add('animate-fadeIn');
        }, 700);
        
        // Form elements animation
        formElements.forEach((el, index) => {
            setTimeout(() => {
                el.style.opacity = '1';
                el.classList.add('animate-fadeIn');
            }, 900 + (index * 150));
        });
    }
    
    // Ripple effect for buttons
    function addRippleEffect() {
        const buttons = document.querySelectorAll('.login-btn, .forgot-password');
        
        buttons.forEach(button => {
            button.addEventListener('click', function(e) {
                const x = e.clientX - e.target.getBoundingClientRect().left;
                const y = e.clientY - e.target.getBoundingClientRect().top;
                
                const ripple = document.createElement('span');
                ripple.classList.add('ripple');
                ripple.style.left = `${x}px`;
                ripple.style.top = `${y}px`;
                
                this.appendChild(ripple);
                
                setTimeout(() => {
                    ripple.remove();
                }, 600);
            });
        });
    }
    
    // Notification function
    function showNotification(message, type = 'success') {
        const notificationIcon = notification.querySelector('.notification-icon i');
        const notificationMessage = notification.querySelector('.notification-message');
        
        // Set message
        notificationMessage.textContent = message;
        
        // Set icon based on type
        if (type === 'success') {
            notificationIcon.className = 'fas fa-check-circle';
            notificationIcon.style.color = '#4CAF50';
        } else if (type === 'error') {
            notificationIcon.className = 'fas fa-times-circle';
            notificationIcon.style.color = '#F44336';
        } else if (type === 'warning') {
            notificationIcon.className = 'fas fa-exclamation-triangle';
            notificationIcon.style.color = '#FF9800';
        } else if (type === 'info') {
            notificationIcon.className = 'fas fa-info-circle';
            notificationIcon.style.color = '#2196F3';
        }
        
        // Show notification
        notification.classList.add('show');
        
        // Hide after 3 seconds
        setTimeout(() => {
            notification.classList.remove('show');
        }, 3000);
    }
    
    // Check for existing session
    function checkSession() {
        const token = localStorage.getItem('auth_token');
        if (token) {
            window.location.href = 'dashboard.html';
        }
    }
    
    // Call session check on page load
    checkSession();
});