// Mobile Dashboard JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // Check Authentication
    if (!Auth.isAuthenticated()) {
        window.location.href = 'index.html';
        return;
    }
    
    // Elements
    const mobileNavItems = document.querySelectorAll('.nav-item');
    const contentSections = document.querySelectorAll('.content-section');
    const notificationBell = document.querySelector('.notification-bell');
    const notificationPanel = document.getElementById('notificationPanel');
    const closeNotifications = document.getElementById('closeNotifications');
    const fabButton = document.querySelector('.fab-button');
    const fabContainer = document.querySelector('.fab-container');
    const fabOptions = document.querySelectorAll('.fab-option');
    const logoutMenuItem = document.getElementById('logoutMenuItem');
    const financeTabs = document.querySelectorAll('.finance-tab');
    const eventTabs = document.querySelectorAll('.event-tab');
    const loaderOverlay = document.querySelector('.loader-overlay');
    
    // Show loader
    loaderOverlay.classList.add('show');
    
    // Setup User Info
    setupUserInfo();
    
    // Setup current date
    setupCurrentDate();
    
    // Fetch Dashboard Data
    fetchDashboardData();
    
    // Initialize Charts
    initializeCharts();
    
    // Event Listeners
    // Bottom Navigation
    mobileNavItems.forEach(item => {
        item.addEventListener('click', function() {
            const sectionId = this.getAttribute('data-section');
            switchSection(sectionId);
        });
    });
    
    // Notification Panel
    notificationBell.addEventListener('click', function() {
        notificationPanel.classList.add('active');
    });
    
    closeNotifications.addEventListener('click', function() {
        notificationPanel.classList.remove('active');
    });
    
    // FAB Button
    fabButton.addEventListener('click', function() {
        fabContainer.classList.toggle('active');
    });
    
    fabOptions.forEach(option => {
        option.addEventListener('click', function() {
            const action = this.getAttribute('data-action');
            handleFabAction(action);
            fabContainer.classList.remove('active');
        });
    });
    
    // Finance Tabs
    financeTabs.forEach(tab => {
        tab.addEventListener('click', function() {
            const period = this.getAttribute('data-period');
            switchFinancePeriod(period);
        });
    });
    
    // Event Tabs
    eventTabs.forEach(tab => {
        tab.addEventListener('click', function() {
            const type = this.getAttribute('data-type');
            switchEventType(type);
        });
    });
    
    // Logout
    logoutMenuItem.addEventListener('click', logout);
    
    // Close notification panel when clicking outside
    document.addEventListener('click', function(e) {
        if (notificationPanel.classList.contains('active') && 
            !notificationPanel.contains(e.target) && 
            !notificationBell.contains(e.target)) {
            notificationPanel.classList.remove('active');
        }
        
        if (fabContainer.classList.contains('active') && 
            !fabContainer.contains(e.target)) {
            fabContainer.classList.remove('active');
        }
    });
    
    // Setup swipe gestures for tab navigation
    setupSwipeNavigation();
    
    // Functions
    function switchSection(sectionId) {
        // Update navigation
        mobileNavItems.forEach(item => {
            if (item.getAttribute('data-section') === sectionId) {
                item.classList.add('active');
            } else {
                item.classList.remove('active');
            }
        });
        
        // Update content sections
        contentSections.forEach(section => {
            if (section.id === sectionId + '-section') {
                section.classList.add('active');
            } else {
                section.classList.remove('active');
            }
        });
    }
    
    function setupUserInfo() {
        const userData = Auth.getUserData();
        
        if (userData) {
            // Set user name
            const userNameElement = document.getElementById('mobileUserName');
            if (userNameElement) {
                userNameElement.textContent = userData.username || 'User';
            }
            
            // Set user initials
            const initials = Utils.getInitials(userData.username || 'User');
            const userInitialsElement = document.getElementById('mobileUserInitials');
            if (userInitialsElement) {
                userInitialsElement.textContent = initials;
            }
        }
    }
    
    function setupCurrentDate() {
        const currentDateElement = document.getElementById('mobileCurrentDate');
        const today = new Date();
        const options = { weekday: 'long', day: 'numeric', month: 'long' };
        currentDateElement.textContent = today.toLocaleDateString(undefined, options);
    }
    
    async function fetchDashboardData() {
        try {
            const response = await fetch('https://edmapi.ed5.in/api/dashboard.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': Auth.getToken()
                }
            });
            
            const data = await response.json();
            
            if (data.status) {
                // Update dashboard with API data
                updateDashboardData(data.data);
            } else {
                showToast('Failed to load dashboard data', 'error');
            }
        } catch (error) {
            console.error('Dashboard data error:', error);
            showToast('Error loading dashboard data', 'error');
        } finally {
            // Hide loader after data is loaded or on error
            setTimeout(() => {
                loaderOverlay.classList.remove('show');
            }, 500);
        }
    }
    
    function updateDashboardData(data) {
        // Quick Stats
        updateQuickStats(data);
        
        // Attendance
        updateAttendance(data.attendance);
        
        // Fee Collection
        updateFeeCollection(data.fee_collection);
        
        // Financial Data
        updateFinancialData(data.financial);
        
        // Students Data
        updateStudentsData(data.students);
        
        // Birthdays
        updateBirthdays(data.birthdays);
        
        // Events
        updateEvents(data.upcoming_events);
        
        // Library Data
        updateLibraryData(data.library);
        
        // Recent Activities
        updateRecentActivities(data.recent_activities);
    }
    
    function updateQuickStats(data) {
        // Total Students
        if (data.students) {
            const totalStudentsElements = document.querySelectorAll('#totalStudents, #totalStudentsCount');
            totalStudentsElements.forEach(element => {
                if (element) element.textContent = Utils.formatNumber(data.students.total_students || 0);
            });
        }
        
        // Total Teachers/Employees
        const totalTeachersElement = document.getElementById('totalTeachers');
        if (totalTeachersElement && data.attendance && data.attendance.employee) {
            totalTeachersElement.textContent = Utils.formatNumber(data.attendance.employee.total || 0);
        }
        
        // Attendance Rate - Use student attendance percentage
        const attendanceRateElement = document.getElementById('attendanceRate');
        if (attendanceRateElement && data.attendance && data.attendance.student) {
            attendanceRateElement.textContent = (data.attendance.student.percentage || 0) + '%';
        }
        
        // Total Revenue/Fee Collection
        const totalRevenueElement = document.getElementById('totalRevenue');
        if (totalRevenueElement && data.fee_collection && data.fee_collection.session) {
            totalRevenueElement.textContent = Utils.formatCurrency(data.fee_collection.session.paid || 0);
        }
    }
    
    function updateAttendance(attendanceData) {
        if (!attendanceData) return;
        
        // Student Attendance
        if (attendanceData.student) {
            const student = attendanceData.student;
            
            // Update percentage
            const studentPercentElement = document.getElementById('studentAttendancePercent');
            if (studentPercentElement) {
                studentPercentElement.textContent = (student.percentage || 0) + '%';
            }
            
            // Update progress bar
            const studentBarElement = document.getElementById('studentAttendanceBar');
            if (studentBarElement) {
                studentBarElement.style.width = (student.percentage || 0) + '%';
            }
            
            // Update counts
            document.getElementById('studentsPresent').textContent = Utils.formatNumber(student.present || 0);
            document.getElementById('studentsAbsent').textContent = Utils.formatNumber(student.absent || 0);
            document.getElementById('studentsLeave').textContent = Utils.formatNumber(student.leave || 0);
            document.getElementById('studentsTotal').textContent = Utils.formatNumber(student.total || 0);
        }
        
        // Staff/Employee Attendance
        if (attendanceData.employee) {
            const staff = attendanceData.employee;
            
            // Update percentage
            const staffPercentElement = document.getElementById('staffAttendancePercent');
            if (staffPercentElement) {
                staffPercentElement.textContent = (staff.percentage || 0) + '%';
            }
            
            // Update progress bar
            const staffBarElement = document.getElementById('staffAttendanceBar');
            if (staffBarElement) {
                staffBarElement.style.width = (staff.percentage || 0) + '%';
            }
            
            // Update counts
            document.getElementById('staffPresent').textContent = Utils.formatNumber(staff.present || 0);
            document.getElementById('staffAbsent').textContent = Utils.formatNumber(staff.absent || 0);
            document.getElementById('staffLeave').textContent = Utils.formatNumber(staff.leave || 0);
            document.getElementById('staffTotal').textContent = Utils.formatNumber(staff.total || 0);
        }
    }
    
    function updateFeeCollection(feeData) {
        if (!feeData) return;
        
        // Session fee data
        if (feeData.session) {
            const session = feeData.session;
            
            // Update total collected and pending
            const feeTotalElement = document.querySelector('.fee-total h3');
            if (feeTotalElement) {
                feeTotalElement.textContent = Utils.formatCurrency(session.paid || 0);
            }
            
            const feePendingElement = document.querySelector('.fee-pending h3');
            if (feePendingElement) {
                feePendingElement.textContent = Utils.formatCurrency(session.pending || 0);
            }
            
            // Update progress
            const feePercentageElement = document.getElementById('feePercentage');
            if (feePercentageElement) {
                feePercentageElement.textContent = (session.percentage || 0).toFixed(2) + '%';
            }
            
            const feeBarElement = document.getElementById('feeCollectionBar');
            if (feeBarElement) {
                feeBarElement.style.width = (session.percentage || 0) + '%';
            }
        }
        
        // Status breakdown
        if (feeData.status_breakdown) {
            const status = feeData.status_breakdown;
            
            document.getElementById('feePaid').textContent = Utils.formatNumber(status.paid || 0);
            document.getElementById('feePartial').textContent = Utils.formatNumber(status.partially_paid || 0);
            document.getElementById('feePending').textContent = Utils.formatNumber(status.pending || 0);
            document.getElementById('feeOverdue').textContent = Utils.formatNumber(status.overdue || 0);
        }
    }
    
    function updateFinancialData(financialData) {
        if (!financialData) return;
        
        // Default to monthly data
        const financeData = financialData.month;
        
        // Update income, expense, balance
        document.getElementById('financeIncome').textContent = Utils.formatCurrency(financeData.income || 0);
        document.getElementById('financeExpense').textContent = Utils.formatCurrency(financeData.expense || 0);
        document.getElementById('financeBalance').textContent = Utils.formatCurrency(financeData.balance || 0);
        
        // Update finance chart with monthly trend
        if (financialData.monthly_trend && window.financeChart) {
            const labels = financialData.monthly_trend.map(item => item.month);
            const incomeData = financialData.monthly_trend.map(item => item.income);
            const expenseData = financialData.monthly_trend.map(item => item.expense);
            
            window.financeChart.data.labels = labels;
            window.financeChart.data.datasets[0].data = incomeData;
            window.financeChart.data.datasets[1].data = expenseData;
            window.financeChart.update();
        }
        
        // Update expense chart
        if (financialData.category_breakdown && financialData.category_breakdown.expense && window.expenseChart) {
            const categories = financialData.category_breakdown.expense.map(item => item.category);
            const amounts = financialData.category_breakdown.expense.map(item => item.amount);
            
            window.expenseChart.data.labels = categories;
            window.expenseChart.data.datasets[0].data = amounts;
            window.expenseChart.update();
        }
    }
    
    function updateStudentsData(studentsData) {
        if (!studentsData) return;
        
        // Update new admissions
        const newAdmissionsElement = document.getElementById('newAdmissionsCount');
        if (newAdmissionsElement) {
            newAdmissionsElement.textContent = Utils.formatNumber(studentsData.new_admissions || 0);
        }
        
        // Update gender distribution
        if (studentsData.gender_distribution) {
            const gender = studentsData.gender_distribution;
            
            document.getElementById('maleCount').textContent = Utils.formatNumber(gender.male || 0);
            document.getElementById('femaleCount').textContent = Utils.formatNumber(gender.female || 0);
        }
        
        // Update course distribution chart
        if (studentsData.course_distribution && window.courseDistributionChart) {
            const courses = studentsData.course_distribution.map(item => item.course);
            const counts = studentsData.course_distribution.map(item => item.count);
            
            window.courseDistributionChart.data.labels = courses;
            window.courseDistributionChart.data.datasets[0].data = counts;
            window.courseDistributionChart.update();
        }
        
        // Update batch distribution chart
        if (studentsData.batch_distribution && window.batchDistributionChart) {
            const batches = studentsData.batch_distribution.map(item => item.batch);
            const counts = studentsData.batch_distribution.map(item => item.count);
            
            window.batchDistributionChart.data.labels = batches;
            window.batchDistributionChart.data.datasets[0].data = counts;
            window.batchDistributionChart.update();
        }
    }
    
    function updateBirthdays(birthdaysData) {
        if (!birthdaysData) return;
        
        const birthdaysContainer = document.getElementById('todayBirthdaysContainer');
        if (!birthdaysContainer) return;
        
        // Clear container
        birthdaysContainer.innerHTML = '';
        
        // Check if there are today's birthdays
        if (birthdaysData.today && 
            ((birthdaysData.today.students && birthdaysData.today.students.length > 0) || 
             (birthdaysData.today.employees && birthdaysData.today.employees.length > 0))) {
            
            // Add students birthdays
            if (birthdaysData.today.students && birthdaysData.today.students.length > 0) {
                birthdaysData.today.students.forEach(student => {
                    const birthdayCard = createBirthdayCard(student, 'student');
                    birthdaysContainer.appendChild(birthdayCard);
                });
            }
            
            // Add employees birthdays
            if (birthdaysData.today.employees && birthdaysData.today.employees.length > 0) {
                birthdaysData.today.employees.forEach(employee => {
                    const birthdayCard = createBirthdayCard(employee, 'employee');
                    birthdaysContainer.appendChild(birthdayCard);
                });
            }
        } else {
            // Hide section if no birthdays today
            const birthdaysSection = document.getElementById('birthdaysSection');
            if (birthdaysSection) {
                birthdaysSection.style.display = 'none';
            }
        }
    }
    
    function createBirthdayCard(person, type) {
        const card = document.createElement('div');
        card.className = 'birthday-card';
        
        // Calculate age
        const birthDate = new Date(person.date_of_birth);
        const today = new Date();
        let age = today.getFullYear() - birthDate.getFullYear();
        
        // Create card HTML
        card.innerHTML = `
            <div class="birthday-avatar">
                <i class="${type === 'student' ? 'fas fa-user-graduate' : 'fas fa-user-tie'}"></i>
            </div>
            <div class="birthday-info">
                <h4>${person.name}</h4>
                <p>${type === 'student' ? `Class ${person.batch || '-'}` : 'Staff'} • ${age} years</p>
            </div>
            <div class="birthday-cake">
                <i class="fas fa-birthday-cake"></i>
            </div>
        `;
        
        return card;
    }
    
    function updateEvents(eventsData) {
        if (!eventsData) return;
        
        const eventsContent = document.getElementById('eventsContent');
        if (!eventsContent) return;
        
        // Default to events tab
        populateEventsList(eventsData.events, eventsContent);
        
        // Setup event tab switching
        const eventTabs = document.querySelectorAll('.event-tab');
        eventTabs.forEach(tab => {
            tab.addEventListener('click', function() {
                // Update active tab
                eventTabs.forEach(t => t.classList.remove('active'));
                this.classList.add('active');
                
                // Update content based on selected tab
                const type = this.getAttribute('data-type');
                if (type === 'events') {
                    populateEventsList(eventsData.events, eventsContent);
                } else if (type === 'exams') {
                    populateEventsList(eventsData.exams, eventsContent);
                }
            });
        });
    }
    
    function populateEventsList(events, container) {
        // Clear container
        container.innerHTML = '';
        
        if (!events || events.length === 0) {
            container.innerHTML = '<p class="no-data">No upcoming events.</p>';
            return;
        }
        
        // Add events
        events.forEach(event => {
            const eventItem = document.createElement('div');
            eventItem.className = 'event-item';
            
            // Get date parts
            const eventDate = new Date(event.start_date || event.date);
            const day = eventDate.getDate();
            const month = eventDate.toLocaleString('default', { month: 'short' });
            
            eventItem.innerHTML = `
                <div class="event-date">
                    <span class="day">${day}</span>
                    <span class="month">${month}</span>
                </div>
                <div class="event-details">
                    <h4>${event.title || event.exam_name}</h4>
                    <p>${event.description || event.subject_name || ''}</p>
                    ${event.start_time ? `
                        <div class="event-time">
                            <i class="far fa-clock"></i>
                            <span>${formatTime(event.start_time)} - ${formatTime(event.end_time)}</span>
                        </div>
                    ` : ''}
                    ${event.venue ? `
                        <div class="event-venue">
                            <i class="fas fa-map-marker-alt"></i>
                            <span>${event.venue}</span>
                        </div>
                    ` : ''}
                </div>
            `;
            
            container.appendChild(eventItem);
        });
    }
    
    function formatTime(timeStr) {
        if (!timeStr) return '';
        
        // Parse time string (HH:MM:SS)
        const [hours, minutes] = timeStr.split(':');
        const hour = parseInt(hours, 10);
        const minute = parseInt(minutes, 10);
        
        // Create Date object and format time
        const date = new Date();
        date.setHours(hour, minute, 0);
        
        return date.toLocaleString('en-US', { 
            hour: 'numeric', 
            minute: '2-digit', 
            hour12: true 
        });
    }
    
    function updateLibraryData(libraryData) {
        if (!libraryData) return;
        
        // Update library stats
        document.getElementById('totalBooks').textContent = libraryData.total_books || 0;
        document.getElementById('issuedBooks').textContent = libraryData.total_issued || 0;
        document.getElementById('availableBooks').textContent = libraryData.total_available || 0;
        document.getElementById('overdueBooks').textContent = libraryData.overdue_books || 0;
        
        // Update recent issues
        const recentIssuesContainer = document.getElementById('recentIssues');
        if (recentIssuesContainer && libraryData.recent_issues && libraryData.recent_issues.length > 0) {
            recentIssuesContainer.innerHTML = '';
            
            libraryData.recent_issues.forEach(issue => {
                const issueItem = document.createElement('div');
                issueItem.className = 'issue-item';
                
                const dueDate = new Date(issue.due_date);
                const isOverdue = dueDate < new Date();
                
                issueItem.innerHTML = `
                    <div class="issue-book">
                        <i class="fas fa-book"></i>
                    </div>
                    <div class="issue-details">
                        <h4>${issue.book_title}</h4>
                        <p>Issued to: ${issue.student_name}</p>
                        <div class="issue-date ${isOverdue ? 'overdue' : ''}">
                            <i class="far fa-calendar-alt"></i>
                            <span>Due: ${Utils.formatDate(issue.due_date)}</span>
                        </div>
                    </div>
                `;
                
                recentIssuesContainer.appendChild(issueItem);
            });
        } else if (recentIssuesContainer) {
            recentIssuesContainer.innerHTML = '<p class="no-data">No recent issues.</p>';
        }
    }
    
    function updateRecentActivities(activitiesData) {
        if (!activitiesData || activitiesData.length === 0) return;
        
        const recentActivitiesContainer = document.getElementById('recentActivities');
        if (!recentActivitiesContainer) return;
        
        // Clear container
        recentActivitiesContainer.innerHTML = '';
        
        // Add recent activities
        activitiesData.forEach(activity => {
            const activityItem = document.createElement('div');
            activityItem.className = `activity-item activity-${activity.type}`;
            
            // Format timestamp
            const timestamp = new Date(activity.timestamp);
            const timeAgo = getTimeAgo(timestamp);
            
            // Format description
            let description = '';
            if (activity.type === 'import') {
                description = `Imported ${activity.description} data`;
            } else {
                description = activity.description;
            }
            
            activityItem.innerHTML = `
                <div class="activity-icon">
                    <i class="fas ${getActivityIcon(activity.type)}"></i>
                </div>
                <div class="activity-details">
                    <p>${description}</p>
                    <div class="activity-time">${timeAgo}</div>
                </div>
            `;
            
            recentActivitiesContainer.appendChild(activityItem);
        });
    }
    
    function getActivityIcon(type) {
        switch (type) {
            case 'import': return 'fa-file-import';
            case 'attendance': return 'fa-clipboard-check';
            case 'fee': return 'fa-money-bill-wave';
            case 'exam': return 'fa-graduation-cap';
            default: return 'fa-history';
        }
    }
    
    function getTimeAgo(date) {
        const now = new Date();
        const diffMs = now - date;
        const diffSec = Math.floor(diffMs / 1000);
        const diffMin = Math.floor(diffSec / 60);
        const diffHour = Math.floor(diffMin / 60);
        const diffDay = Math.floor(diffHour / 24);
        
        if (diffDay > 0) {
            return diffDay === 1 ? 'Yesterday' : `${diffDay} days ago`;
        } else if (diffHour > 0) {
            return `${diffHour} ${diffHour === 1 ? 'hour' : 'hours'} ago`;
        } else if (diffMin > 0) {
            return `${diffMin} ${diffMin === 1 ? 'minute' : 'minutes'} ago`;
        } else {
            return 'Just now';
        }
    }
    
    function switchFinancePeriod(period) {
        // Update active tab
        financeTabs.forEach(tab => {
            if (tab.getAttribute('data-period') === period) {
                tab.classList.add('active');
            } else {
                tab.classList.remove('active');
            }
        });
        
        // The actual data switching is handled in the updateFinancialData function
        // For demo purposes, we'll just simulate it by fetching the dashboard data again
        loaderOverlay.classList.add('show');
        
        // Simulate API call delay
        setTimeout(() => {
            fetchDashboardData();
        }, 500);
    }
    
    function switchEventType(type) {
        // Update active tab
        eventTabs.forEach(tab => {
            if (tab.getAttribute('data-type') === type) {
                tab.classList.add('active');
            } else {
                tab.classList.remove('active');
            }
        });
        
        // The actual content switching is handled in the updateEvents function
    }
    
    function handleFabAction(action) {
        switch (action) {
            case 'new-student':
                showToast('New student admission form opened');
                break;
            case 'new-event':
                showToast('New event creation form opened');
                break;
            case 'take-attendance':
                showToast('Attendance marking page opened');
                break;
            default:
                break;
        }
    }
    
    function logout() {
        // Show loader
        loaderOverlay.classList.add('show');
        
        // Simulate API call
        setTimeout(() => {
            Auth.logout();
        }, 1000);
    }
    
    function setupSwipeNavigation() {
        // Only setup if Hammer.js is available
        if (typeof Hammer !== 'undefined') {
            // Get main content element
            const contentElement = document.querySelector('.mobile-content');
            
            // Initialize Hammer
            const hammer = new Hammer(contentElement);
            
            // Configure horizontal swipe
            hammer.get('swipe').set({ direction: Hammer.DIRECTION_HORIZONTAL });
            
            // Handle swipe events
            hammer.on('swipeleft swiperight', function(e) {
                // Get current active section
                const activeSection = document.querySelector('.content-section.active');
                const activeSectionId = activeSection.id.replace('-section', '');
                
                // Get current index
                const sectionIds = ['dashboard', 'students', 'stats', 'events', 'more'];
                const currentIndex = sectionIds.indexOf(activeSectionId);
                
                if (currentIndex === -1) return;
                
                // Calculate new index based on swipe direction
                let newIndex;
                if (e.type === 'swipeleft') {
                    // Next section (swipe left)
                    newIndex = Math.min(currentIndex + 1, sectionIds.length - 1);
                } else {
                    // Previous section (swipe right)
                    newIndex = Math.max(currentIndex - 1, 0);
                }
                
                // Switch to new section if different
                if (newIndex !== currentIndex) {
                    switchSection(sectionIds[newIndex]);
                }
            });
        }
    }
    
    function initializeCharts() {
        // Finance Chart (Monthly Income/Expense)
        const financeChartCtx = document.getElementById('financeChart');
        if (financeChartCtx) {
            window.financeChart = new Chart(financeChartCtx, {
                type: 'line',
                data: {
                    labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'],
                    datasets: [
                        {
                            label: 'Income',
                            data: [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
                            borderColor: '#11998e',
                            backgroundColor: 'rgba(17, 153, 142, 0.1)',
                            tension: 0.4,
                            fill: true
                        },
                        {
                            label: 'Expense',
                            data: [0, 0, 17700, 0, 0, 0, 0, 0, 0, 0, 0, 0],
                            borderColor: '#f12711',
                            backgroundColor: 'rgba(241, 39, 17, 0.1)',
                            tension: 0.4,
                            fill: true
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'top',
                            labels: {
                                boxWidth: 12,
                                usePointStyle: true,
                                pointStyle: 'circle'
                            }
                        },
                        tooltip: {
                            mode: 'index',
                            intersect: false,
                            callbacks: {
                                label: function(context) {
                                    return context.dataset.label + ': ₹' + context.raw.toLocaleString();
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                callback: function(value) {
                                    return '₹' + value.toLocaleString();
                                }
                            }
                        }
                    }
                }
            });
        }
        
        // Course Distribution Chart
        const courseChartCtx = document.getElementById('courseDistributionChart');
        if (courseChartCtx) {
            window.courseDistributionChart = new Chart(courseChartCtx, {
                type: 'doughnut',
                data: {
                    labels: ['STD I', 'STD II', 'STD III', 'STD IV', 'STD V', 'STD VI', 'STD VII', 'STD VIII', 'STD IX', 'STD X', 'Others'],
                    datasets: [{
                        data: [108, 100, 100, 100, 74, 86, 85, 90, 90, 90, 141],
                        backgroundColor: [
                            '#6a11cb', '#2575fc', '#FF512F', '#DD2476', 
                            '#11998e', '#38ef7d', '#f5af19', '#f12711',
                            '#0082c8', '#667eea', '#7F00FF'
                        ],
                        borderWidth: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: {
                                boxWidth: 12,
                                usePointStyle: true,
                                pointStyle: 'circle'
                            }
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    const label = context.label || '';
                                    const value = context.raw || 0;
                                    const total = context.dataset.data.reduce((acc, curr) => acc + curr, 0);
                                    const percentage = Math.round((value / total) * 100);
                                    return `${label}: ${value} (${percentage}%)`;
                                }
                            }
                        }
                    }
                }
            });
        }
        
        // Batch Distribution Chart
        const batchChartCtx = document.getElementById('batchDistributionChart');
        if (batchChartCtx) {
            window.batchDistributionChart = new Chart(batchChartCtx, {
                type: 'pie',
                data: {
                    labels: ['A', 'B', 'C', 'D', 'Science'],
                    datasets: [{
                        data: [329, 320, 299, 100, 16],
                        backgroundColor: [
                            '#6a11cb', '#2575fc', '#FF512F', '#DD2476', '#11998e'
                        ],
                        borderWidth: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: {
                                boxWidth: 12,
                                usePointStyle: true,
                                pointStyle: 'circle'
                            }
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    const label = context.label || '';
                                    const value = context.raw || 0;
                                    const total = context.dataset.data.reduce((acc, curr) => acc + curr, 0);
                                    const percentage = Math.round((value / total) * 100);
                                    return `${label}: ${value} (${percentage}%)`;
                                }
                            }
                        }
                    }
                }
            });
        }
        
        // Expense Chart
        const expenseChartCtx = document.getElementById('expenseChart');
        if (expenseChartCtx) {
            window.expenseChart = new Chart(expenseChartCtx, {
                type: 'bar',
                data: {
                    labels: ['Electricity', 'Newspaper'],
                    datasets: [{
                        label: 'Expense Amount',
                        data: [15000, 2700],
                        backgroundColor: [
                            'rgba(106, 17, 203, 0.7)',
                            'rgba(241, 39, 17, 0.7)'
                        ],
                        borderColor: [
                            '#6a11cb',
                            '#f12711'
                        ],
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return `Amount: ₹${context.raw.toLocaleString()}`;
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                callback: function(value) {
                                    return '₹' + value.toLocaleString();
                                }
                            }
                        }
                    }
                }
            });
        }
        
        // Initialize Calendar
        initializeCalendar();
    }
    
    function initializeCalendar() {
        const calendarDays = document.getElementById('calendarDays');
        const currentMonthElement = document.getElementById('currentMonth');
        const prevMonthButton = document.getElementById('prevMonth');
        const nextMonthButton = document.getElementById('nextMonth');
        
        if (!calendarDays || !currentMonthElement) return;
        
        // Calendar state
        let currentDate = new Date();
        let selectedDate = new Date();
        
        // Event data (to be populated from API)
        let eventDates = [];
        
        // Initial calendar render
        renderCalendar(currentDate);
        
        // Event listeners for month navigation
        if (prevMonthButton) {
            prevMonthButton.addEventListener('click', function() {
                currentDate.setMonth(currentDate.getMonth() - 1);
                renderCalendar(currentDate);
            });
        }
        
        if (nextMonthButton) {
            nextMonthButton.addEventListener('click', function() {
                currentDate.setMonth(currentDate.getMonth() + 1);
                renderCalendar(currentDate);
            });
        }
        
        // Function to render calendar
        function renderCalendar(date) {
            // Update month/year display
            const monthYearStr = date.toLocaleString('default', { month: 'long', year: 'numeric' });
            currentMonthElement.textContent = monthYearStr;
            
            // Clear calendar days
            calendarDays.innerHTML = '';
            
            // Get first day of month and total days
            const firstDay = new Date(date.getFullYear(), date.getMonth(), 1);
            const lastDay = new Date(date.getFullYear(), date.getMonth() + 1, 0);
            const totalDays = lastDay.getDate();
            
            // Get day of week for first day (0 = Sunday, 6 = Saturday)
            const firstDayIndex = firstDay.getDay();
            
            // Get previous month's last days
            const prevLastDay = new Date(date.getFullYear(), date.getMonth(), 0).getDate();
            
            // Calculate days needed from next month
            const nextDays = 42 - (firstDayIndex + totalDays); // 42 = 6 rows * 7 days
            
            // Add previous month's days
            for (let i = firstDayIndex - 1; i >= 0; i--) {
                const dayElement = document.createElement('div');
                dayElement.className = 'day inactive';
                dayElement.textContent = prevLastDay - i;
                calendarDays.appendChild(dayElement);
            }
            
            // Add current month's days
            const today = new Date();
            for (let i = 1; i <= totalDays; i++) {
                const dayElement = document.createElement('div');
                dayElement.className = 'day';
                dayElement.textContent = i;
                
                // Check if it's today
                if (date.getFullYear() === today.getFullYear() && 
                    date.getMonth() === today.getMonth() && 
                    i === today.getDate()) {
                    dayElement.classList.add('today');
                }
                
                // Check if it's selected date
                if (date.getFullYear() === selectedDate.getFullYear() && 
                    date.getMonth() === selectedDate.getMonth() && 
                    i === selectedDate.getDate()) {
                    dayElement.classList.add('selected');
                }
                
                // Check if day has events (demo)
                const currentDateStr = `${date.getFullYear()}-${(date.getMonth() + 1).toString().padStart(2, '0')}-${i.toString().padStart(2, '0')}`;
                if (eventDates.includes(currentDateStr)) {
                    dayElement.classList.add('has-event');
                }
                
                // Add click event
                dayElement.addEventListener('click', function() {
                    // Remove selected class from all days
                    document.querySelectorAll('.day').forEach(day => {
                        day.classList.remove('selected');
                    });
                    
                    // Add selected class to clicked day
                    this.classList.add('selected');
                    
                    // Update selected date
                    selectedDate = new Date(date.getFullYear(), date.getMonth(), i);
                    
                    // Update events for selected date
                    updateSelectedDateEvents(selectedDate);
                });
                
                calendarDays.appendChild(dayElement);
            }
            
            // Add next month's days
            for (let i = 1; i <= nextDays; i++) {
                const dayElement = document.createElement('div');
                dayElement.className = 'day inactive';
                dayElement.textContent = i;
                calendarDays.appendChild(dayElement);
            }
        }
        
        function updateSelectedDateEvents(date) {
            const dayEventsElement = document.getElementById('dayEvents');
            const selectedDateElement = document.getElementById('selectedDate');
            
            if (!dayEventsElement || !selectedDateElement) return;
            
            // Update selected date display
            selectedDateElement.textContent = date.toLocaleDateString('default', { 
                day: 'numeric',
                month: 'long',
                year: 'numeric'
            });
            
            // Demo events - would be populated from API
            const events = [
                {
                    date: '2025-03-17',
                    title: 'SSLC PHY',
                    time: '11:00 AM - 1:00 PM',
                    venue: 'Auditorium'
                },
                {
                    date: '2025-03-17',
                    title: 'Political Science +2',
                    time: '2:00 PM - 5:00 PM',
                    venue: 'Auditorium'
                },
                {
                    date: '2025-03-19',
                    title: 'Accounts +2',
                    time: '2:00 PM - 5:00 PM',
                    venue: 'Auditorium'
                }
            ];
            
            // Format date for comparison
            const dateStr = `${date.getFullYear()}-${(date.getMonth() + 1).toString().padStart(2, '0')}-${date.getDate().toString().padStart(2, '0')}`;
            
            // Filter events for selected date
            const dayEvents = events.filter(event => event.date === dateStr);
            
            // Clear events container
            dayEventsElement.innerHTML = '';
            
            // Add events or no events message
            if (dayEvents.length > 0) {
                dayEvents.forEach(event => {
                    const eventItem = document.createElement('div');
                    eventItem.className = 'event-item';
                    
                    eventItem.innerHTML = `
                        <div class="event-icon">
                            <i class="fas fa-calendar-day"></i>
                        </div>
                        <div class="event-details">
                            <h4>${event.title}</h4>
                            <div class="event-time">
                                <i class="far fa-clock"></i>
                                <span>${event.time}</span>
                            </div>
                            <div class="event-venue">
                                <i class="fas fa-map-marker-alt"></i>
                                <span>${event.venue}</span>
                            </div>
                        </div>
                    `;
                    
                    dayEventsElement.appendChild(eventItem);
                });
            } else {
                dayEventsElement.innerHTML = '<p class="no-data">No events scheduled for this day.</p>';
            }
        }
        
        // Build event dates array from API data
        function setEventDates(events) {
            eventDates = events.map(event => event.date || event.start_date);
            renderCalendar(currentDate); // Re-render to show event indicators
        }
        
        // Demo event dates
        setEventDates([
            { date: '2025-03-17' },
            { date: '2025-03-19' },
            { date: '2025-03-21' }
        ]);
    }
    
    function showToast(message, type = 'success') {
        const toast = document.getElementById('toastNotification');
        const toastIcon = toast.querySelector('.toast-icon i');
        const toastMessage = toast.querySelector('.toast-message');
        const toastClose = toast.querySelector('.toast-close');
        
        // Set message
        toastMessage.textContent = message;
        
        // Set icon and color based on type
        switch (type) {
            case 'success':
                toastIcon.className = 'fas fa-check-circle';
                toast.querySelector('.toast-icon').style.background = '#4CAF50';
                break;
            case 'error':
                toastIcon.className = 'fas fa-times-circle';
                toast.querySelector('.toast-icon').style.background = '#F44336';
                break;
            case 'warning':
                toastIcon.className = 'fas fa-exclamation-triangle';
                toast.querySelector('.toast-icon').style.background = '#FF9800';
                break;
            case 'info':
                toastIcon.className = 'fas fa-info-circle';
                toast.querySelector('.toast-icon').style.background = '#2196F3';
                break;
        }
        
        // Show toast
        toast.classList.add('show');
        
        // Auto hide after 3 seconds
        const toastTimeout = setTimeout(() => {
            toast.classList.remove('show');
        }, 3000);
        
        // Close button
        toastClose.addEventListener('click', function() {
            clearTimeout(toastTimeout);
            toast.classList.remove('show');
        });
    }
});